/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: AtomSel.C,v $
 *      $Author: dalke $        $Locker:  $                $State: Exp $
 *      $Revision: 1.20 $      $Date: 1995/06/29 23:48:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * Parse and maintain the data for selecting atoms.
 *
 ***************************************************************************/
#ifdef ARCH_HPUX9
  static char ident[] = "@(#)$Header: /Home/h2/billh/projects/vmd/src/RCS/AtomSel.C,v 1.20 1995/06/29 23:48:32 dalke Exp $";
#endif

#include <math.h>
#include <stdlib.h>
#include "Global.h"
#include "AtomSel.h"
#include "BaseMolecule.h"
#include "SymbolTable.h"
#include "ParseTree.h"

  
// a string array with text descriptions of selection methods
//char *AtomSelName[AtomSel::TOTAL] = { "None", "All", "Backbone", "Name",
//	"Type", "ResName", "ResType", "ResID", "SegName", "ID" ,
//	"Protein", "Nucleic", "Waters", "Fragment", "PFrag", "NFrag"};



//////
//   These functions get the information from the appropriate
// BaseMolecule, as defined in 'atom_sel_mol
static BaseMolecule *atom_sel_mol;

// 'all'
static int atomsel_all(int num, int *flgs)
{
   for (int i=num-1; i>=0; i--) {
      flgs[i] = TRUE;
   }
   return 1;
}
// 'none'
static int atomsel_none(int num, int *flgs)
{
   for (int i=num-1; i>=0; i--) {
      flgs[i] = FALSE;
   }
   return 1;
}

#define generic_atom_data(fctnname, datatype, field)			      \
static int fctnname(int num, datatype *data, int *flgs)			      \
{									      \
   for (int i=0; i<num; i++) {						      \
      if (flgs[i]) {							      \
	 data[i] = atom_sel_mol -> atom(i)->field;			      \
      }									      \
   }									      \
   return 1;								      \
}

// 'name'
generic_atom_data(atomsel_name, GString, namestr);
// 'type'
generic_atom_data(atomsel_type, GString, typestr);
// 'index'
generic_atom_data(atomsel_index, int, index);
// 'fragment'
generic_atom_data(atomsel_fragment, int, fragment);
// 'numbonds'
generic_atom_data(atomsel_numbonds, int, bonds);
// 'residue'
generic_atom_data(atomsel_residue, int, uniq_resid);
// 'resname'
generic_atom_data(atomsel_resname, GString, resnamestr);
// 'chain'
generic_atom_data(atomsel_chain, GString, chainstr);
// 'segname'
generic_atom_data(atomsel_segname, GString, segnamestr);
// 'radius'
generic_atom_data(atomsel_radius, double, radius());
// 'mass'
generic_atom_data(atomsel_mass, double, mass());
// 'charge'
generic_atom_data(atomsel_charge, double, charge());
// 'beta'
generic_atom_data(atomsel_beta, double, beta());
// 'occupancy?'
generic_atom_data(atomsel_occupancy, double, occup());


// 'resid'
// this is unusual as I convert from string to an int
static int atomsel_resid(int num, int *data, int *flgs)
{
   for (int i=0; i<num; i++) {
      if (flgs[i]) {
	 data[i] = atoi(atom_sel_mol -> atom(i)->residstr);
      }
   }
   return 1;
}


#define generic_atom_boolean(fctnname, comparison)			      \
static int fctnname(int num, int *flgs)					      \
{									      \
   for (int i=0; i<num; i++) {						      \
      if (flgs[i]) {							      \
	 flgs[i] = atom_sel_mol -> atom(i) -> comparison;		      \
      }									      \
   }									      \
   return 1;								      \
}

// 'backbone'
generic_atom_boolean(atomsel_backbone, atomType != Atom::NORMAL);
// 'protein'
generic_atom_boolean(atomsel_protein, residueType == Atom::PROTEIN);
// 'nucleic'
generic_atom_boolean(atomsel_nucleic, residueType == Atom::NUCLEIC);
// 'water'
generic_atom_boolean(atomsel_water, residueType == Atom::WATERS);


//// specialized function to turn on all atoms in a given residue
//      and leave the rest alone.
// It is slower this way, but easier to understand
static void mark_atoms_given_residue(int residue, int *on)
{
  ResizeArray<int> *atoms = &(atom_sel_mol->residueList[residue]->atoms);
  for (int i= atoms->num()-1; i>=0; i--) {
     on[(*atoms)[i]] = TRUE;
  }
}


// macro for either protein or nucleic fragments
#define fragment_data(fctn, fragtype)					      \
static int fctn(int num, int *data, int *)				      \
{									      \
   int *tmp = new int[num];						      \
   for (int i=num-1; i>=0; i--) {  /* clear the arrays */		      \
      tmp[i] = 0;							      \
      data[i] = -1;  /* default is -1 for 'not a [np]frag' */		      \
   }									      \
   /* for each fragment */						      \
   for ( i=atom_sel_mol->fragtype.num()-1; i>=0; i--) {			      \
      /* for each residues of the fragment */				      \
      for (int j=atom_sel_mol->fragtype[i]->num()-1; j>=0; j--) {	      \
	 /* mark the atoms in the fragment */				      \
	 mark_atoms_given_residue((*atom_sel_mol->fragtype[i])[j], tmp);      \
      }									      \
      /* and label them with the appropriate number */			      \
      for (j=num-1; j>=0; j--) {					      \
	 if (tmp[j]) {							      \
	    data[j] = i;						      \
	    tmp[j] = 0;							      \
	 }								      \
      }									      \
   }									      \
   delete [] tmp;							      \
   return 1;								      \
}

fragment_data(atomsel_pfrag, pfragList);
fragment_data(atomsel_nfrag, nfragList);


#define position_data(fctn, offset)					      \
static int fctn(int num, double *data, int *flgs)			      \
{									      \
   if (atom_sel_mol->is_current()) {					      \
      float *r = atom_sel_mol->current()->pos;				      \
      if (!r) return 0; /* just in case, but should never happen */	      \
      for (int i=num-1; i>=0; i--) {					      \
	 if (flgs[i]) {							      \
	    data[i] = r[3*i + offset];					      \
	 }								      \
      }									      \
   }									      \
   return 1;								      \
}

position_data(atomsel_xpos, 0);
position_data(atomsel_ypos, 1);
position_data(atomsel_zpos, 2);

static double atomsel_square(double x)
{
   return x*x;
}
	      
// called in Global.C . This sets up the function pointers for the
// seletion commands in the global variable
void atomSelParser_init(void)
{
   atomSelParser.add_singleword("[Aa][Ll][Ll]", "all", atomsel_all);
   atomSelParser.add_singleword("[Nn][Oo][Nn][Ee]", "none", atomsel_none);
   
   atomSelParser.add_keyword("[Nn][Aa][Mm][Ee]", "name", atomsel_name);
   atomSelParser.add_keyword("[Tt][Yy][Pp][Ee]", "type", atomsel_type);
   
   atomSelParser.add_keyword("[Ii][Nn][Dd][Ee][Xx]", "index", atomsel_index);
   atomSelParser.add_keyword("[Ff][Rr][Aa][Gg][Mm][Ee][Nn][Tt]",
			     "fragment", atomsel_fragment);
   atomSelParser.add_keyword("[Pp][Ff][Rr][Aa][Gg]",
			     "pfrag", atomsel_pfrag);
   atomSelParser.add_keyword("[Nn][Ff][Rr][Aa][Gg]",
			     "nfrag", atomsel_nfrag);
   atomSelParser.add_keyword("[Nn][Uu][Mm][Bb][Oo][Nn][Dd][Ss]",
			     "numbonds", atomsel_numbonds);
   atomSelParser.add_keyword("[Rr][Ee][Ss][Ii][Dd][Uu][Ee]", "residue",
                             atomsel_residue);
   atomSelParser.add_keyword("[Rr][Ee][Ss][Nn][Aa][Mm][Ee]",
			     "resname", atomsel_resname);
   atomSelParser.add_keyword("[Rr][Ee][Ss][Ii][Dd]", "resid", atomsel_resid);
   atomSelParser.add_keyword("[Cc][Hh][Aa][Ii][Nn]",
			     "chain", atomsel_chain);
   atomSelParser.add_keyword("[Ss][Ee][Gg][Nn][Aa][Mm][Ee]",
			     "segname", atomsel_segname);

   atomSelParser.add_singleword("[Bb][Aa][Cc][Kk][Bb][Oo][Nn][Ee]",
			    "backbone", atomsel_backbone);
   atomSelParser.add_singleword("[Pp][Rr][Oo][Tt][Ee][Ii][Nn]",
			    "protein", atomsel_protein);
   atomSelParser.add_singleword("[Nn][Uu][Cc][Ll][Ee][Ii][Cc]",
			    "nucleic", atomsel_nucleic);
   atomSelParser.add_singleword("[Ww][Aa][Tt][Ee][Rr][Ss]?",
			    "water", atomsel_water);


   atomSelParser.add_keyword("x", "x", atomsel_xpos);
   atomSelParser.add_keyword("y", "y", atomsel_ypos);
   atomSelParser.add_keyword("z", "z", atomsel_zpos);
   atomSelParser.add_keyword("radius", "radius", atomsel_radius);
   atomSelParser.add_keyword("mass", "mass", atomsel_mass);
   atomSelParser.add_keyword("charge", "charge", atomsel_charge);
   atomSelParser.add_keyword("beta", "beta", atomsel_beta);
   atomSelParser.add_keyword("occupancy", "occupancy", atomsel_occupancy);
   
   // and a few functions for good measure
   atomSelParser.add_function("sqr", "sqr", atomsel_square);
   atomSelParser.add_function("sqrt", "sqrt", sqrt);
   atomSelParser.add_function("abs", "abs", fabs);
   atomSelParser.add_function("floor", "floor", floor);
   atomSelParser.add_function("ceil", "ceil", ceil);
//   atomSelParser.add_function("trunc", "trunc", trunc);
   atomSelParser.add_function("rint", "rint", rint);
   
			      
   atomSelParser.add_function("sin", "sin", sin);
   atomSelParser.add_function("cos", "cos", cos);
   atomSelParser.add_function("tan", "tan", tan);
   atomSelParser.add_function("atan", "atan", atan);
   atomSelParser.add_function("asin", "asin", asin);
   atomSelParser.add_function("acos", "acos", acos);

   atomSelParser.add_function("sinh", "sinh", sinh);
   atomSelParser.add_function("cosh", "cosh", cosh);
   atomSelParser.add_function("tanh", "tanh", tanh);

   atomSelParser.add_function("exp", "exp", exp);
   atomSelParser.add_function("log", "log", log);
   atomSelParser.add_function("log10", "log10", log10);
   atomSelParser.add_function("erf", "erf", erf);
   atomSelParser.add_function("erfc", "erfc", erfc);
}

//////////////////////////  constructor and destructor
// constructor; parse string and see if OK
AtomSel::AtomSel(MoleculeList *mlist) {
  
  // initialize variables
  molList = mlist;
  mol = NULL;
  selected = NO_PARSE;
  on = NULL;
  cmdStr = NULL;
  tree = NULL;
  change(DEFAULT_ATOMSEL);
}


// copy constructor

AtomSel::AtomSel(AtomSel& as) {
   cmdStr = NULL;
   tree = NULL;
   on = NULL;
   selected = NO_PARSE;
   mol = as.mol;
   molList = as.molList;
   if (as.cmdStr) {       // now make the parse tree
      change(as.cmdStr);
   } else {
      change(DEFAULT_ATOMSEL);
   }
   find(mol);       // and find the data
  
}


// destructor; free up space
AtomSel::~AtomSel(void) {
   if(on) {
      delete [] on;
   }
   if (tree) {
      delete tree;
   }
   if (cmdStr) {
      delete [] cmdStr;
   }
}

// assignment operator, to change the current settings.
// The BaseMolecule gets the AtomSel for the whole system.
// That means it does NOT change the current molecule.
AtomSel& AtomSel::operator=(const AtomSel &as) {
  if (tree) {
     delete tree;
     tree = NULL;
  }
  if (on) {
     delete [] on;
     on = NULL;
  }
  change(as.cmdStr);
  return *this;
}


// provide new settings; does a 'find' at the end if a mol has
// been previously provided.  Return the number of atoms found
// or ATOMSEL_NO_PARSE if bad text
// or ATOMSEL_NO_MOL if no molecule
int AtomSel::change(char *newcmd)
{
   ParseTree *newtree = atomSelParser.parse(newcmd);
   if (!newtree) {
      return NO_PARSE;
   }
   if (cmdStr) delete [] cmdStr;
   cmdStr = stringdup(newcmd);
   if (tree) delete tree;
   tree = newtree;
   // and evaluate
   int i = find(mol);
   return i;
}

// find the selection for a molecule
// return the number of atoms found
// or NO_MOL or NO_PARSE if no good
int AtomSel::find(BaseMolecule *m)
{
   if (!tree) {
      return NO_PARSE;
   }
   if (!m) {
      return NO_MOL;
   }
   atom_sel_mol = mol = m;
   if (on) delete [] on;
   on = new int[mol->nAtoms];
   tree->evaluate(mol->nAtoms, on);
   // count up the number of 1s (there are only 0s and 1s)
   // 'selected' is a class variable
   selected = 0;
   for (int i=mol->nAtoms-1; i>=0; i--) {
      selected += on[i];
   }
   return selected;
}

