/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: BaseMolecule.h,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.15 $	$Date: 1995/06/29 23:48:32 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Base class for all molecules, without display-specific information.  This
 * portion of a molecule contains the structural data, and all routines to
 * find the structure (backbone, residues, etc).  It contains the
 * animation list as well.
 *
 ***************************************************************************/
#ifndef BASEMOLECULE_H
#define BASEMOLECULE_H

#include <stdio.h>
#include "Animation.h"
#include "NameList.h"
#include "Atom.h"
#include "Residue.h"
#include "Fragment.h"
#include "ResizeArray.h"

class BaseMolecule : public Animation {

public:
  // different types of molecules
  enum MoleculeType { UAPROTEIN, EHPROTEIN, UAPROTDNA, EHPROTDNA, NUCLEIC,
  	ORGANIC, INORGANIC };

  // type of molecule (from list above)
  MoleculeType type;

  //
  // public molecular structure data (for ease of access):
  //
  
  int nAtoms;			// number of atoms
  int nBonds;			// number of bonds
  int nBackProtein;		// number of protein backbone bonds
  int nBackDNA;			// number of DNA backbone bonds
  int nResidues;		// number of residues
  int nSegments;		// number of segments
  NameList<int> atomNames;	// list of unique atom names
  NameList<int> atomTypes;	// list of unique atom types
  NameList<int> resNames;	// list of unique residue names
  NameList<int> resIds;		// list of unique residue ids (kept as strings)
  NameList<int> chainNames;	// list of unique chain names
  NameList<int> segNames;	// list of unique segment names

  ResizeArray<Residue *> residueList;	// list of which residues are 
  					// connected to which
  ResizeArray<Fragment *> fragList;	// list of connected residues
  ResizeArray<Fragment *> pfragList;	// list of connected protein residues
  					//  this is a single chain from N to C
  ResizeArray<Fragment *> nfragList;	// ditto for nuc; from 5' to 3'

private:
  // static counter for determining what the next "Molecule Number" is
  static int nextIDNumber;

protected:
  // unique ID number, assigned after successful initialization
  int ID;

  //
  // molecular structure data:
  //

  int maxAtoms;				// max number of atoms
  Atom **atomList;			// atom data
  
  //
  // routines to develop molecular structure
  //
  
  // initialize lists, but do not put in any data
  void init_atoms(int);		// initialize atoms list

  // add a new atom; return it's index.
  int add_atom(char *name, char *type, char *resname, char *resid,
  	char *chainid, char *segname, float *pos, float *extra);

  // add a new bond from a to b; return total number of bonds added so far.
  int add_bond(int, int, Atom::BackboneType = Atom::NORMAL);
  
  //
  // routines to determine components of molecular structure
  //
  
  // find the backbone bonds; return number found.
  int find_backbone(void);
    
  // find the residues in the molecule; return number found.
  // I look for atoms with the same resid connected to 
  // backbone atoms with the same resid (find enough backbone
  // atoms, then find atoms connected to them)
  int find_connected_backbone(Atom::BackboneType , int, int , int , int *);
  void clean_up_connection(int , int , int *);
  void find_connected_atoms_in_resid( Atom::ResidueType , int , 
     int , int , int *);
  void find_and_mark(int , Atom::BackboneType, Atom::ResidueType,
     int *, int *);
  void make_uniq_resid(int, int, int, int *); // give each residue a uniq resid
  int find_residues(void);
  
  // Find the waters, based on resname, and return number.
  // This should take place after find_residues to keep
  // from mistaking a protein resname as a water resname, maybe
  void find_connected_waters(int i, int idx, char *tmp);
  int find_waters(void);

  // find which residues are connected to which
  void find_connected_residues(int num_residues);
  
  // find the segments in the molecule; return number found.
  int find_segments(void) { return segNames.num(); }

  
  // find the set of the set of connected residues
  // and put the info in fragList
  void find_connected_fragment(int i, int fragnum, int *flgs);
  int find_fragments(void);
  
  // find ordered protein and nucleic subfragments
  void find_connected_subfragment(int resnum, int fragnum, 
         int *flgs, int endatom,  int altendatom, Atom::ResidueType restype, 
         ResizeArray<Fragment *> *subfragList);
  void find_subfragments(int startatom, int endatom, int altendatom,
    Atom::ResidueType restype, ResizeArray<Fragment *> *subfragList);

public:
  // finds only the first occurance of an atom name in the residue
  // returns -1 if it wasn't found
  int find_atom_in_residue(int atomnameindex, int residue);
  int find_atom_in_residue(char *atomname, int residue);
  

public:
  // constructor; just sets most things to null.  Derived classes must put
  // in structure in 'create' routine.  Typical sequence of creating a
  // molecule should be:
  //	mol = new Molecule(....)
  //	( specify parameters for creation )
  //	mol->create();	... return success
  BaseMolecule(void);
  
  // destructor; free up space allocated by this object
  virtual ~BaseMolecule(void);
  
  // create the molecule.  Successive classes should call their parent 'create'
  // after they are done.
  // Returns success .... if not successful, should be 'delete'ed
  virtual int create(void);
  
  //
  // query info about the molecule
  //
  
  // id number of this molecule
  int id(void) { return ID; }

  // Nth bond, atom, and backbone bond.  All assume correct index and that
  // the structure has been initialized (for speed).
  Atom *atom(int n) { return atomList[n]; }
  
  // return a string containing the full name specification for the
  // specified atom.  If the second argument is not NULL, the string is
  // also copied into the given string storage
  char *atom_full_name(int, char * = NULL);

  // return a string containing a shorter name specification for the
  // specified atom.  If the second argument is not NULL, the string is
  // also copied into the given string storage
  char *atom_short_name(int, char * = NULL);

  // scaling factor required to make the molecule fit within (-1 ... 1)
  virtual float scale_factor(void);
  
  // center of volume of this molecule
  virtual void cov(float &, float &, float &);

  //
  // informative routines
  //
  
  // return 'default' charge, mass, occupancy value and beta value
  float default_charge(char *);
  float default_mass(char *);
  float default_radius(char *);
  float default_occup(void) { return 1.0; }
  float default_beta(void) { return 0.0; }

};

#endif

