/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CaveScene.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.9 $	$Date: 1995/06/07 01:35:52 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   The CAVE specific Scene.  It has to get information from the
 * shared memory arena, since the display lists are shared amoung
 * the different machines
 *
 ***************************************************************************/

#include <stdlib.h>

#include <cave.h>
#include "CaveScene.h"
#include "CaveRoutines.h"
#include "DisplayDevice.h"
#include "Inform.h"
#include "utilities.h"

#define INITLISTSIZE 32
////////////////////////////  constructor  
CaveScene::CaveScene(void)
        : cmdListArray2D(INITLISTSIZE, 2.0), cmdListArray3D(INITLISTSIZE, 2.0),
          dispObjList2D(INITLISTSIZE, 2.0), dispObjList3D(INITLISTSIZE, 2.0) {

  MSGDEBUG(1,"Creating a Normal Scene ..." << sendmsg);
  cave_numDisplayable2D = (int *) malloc_from_CAVE_memory(sizeof(float));
  cave_numDisplayable3D = (int *) malloc_from_CAVE_memory(sizeof(float));
  cave_num_disp_list2D = (int *) malloc_from_CAVE_memory(sizeof(float));
  cave_num_disp_list3D = (int *) malloc_from_CAVE_memory(sizeof(float));
  if (!cave_numDisplayable2D || !cave_numDisplayable3D || 
      !cave_num_disp_list2D || !cave_num_disp_list3D) {
    msgErr << "CANNOT ALLOCATE SHARED MEMORY FOR CaveScene CLASS !!!"
      <<sendmsg;
  }

    // set up the shared flags used to control when the rendering occurs
  draw_flags = (int *) malloc_from_CAVE_memory(
                         sizeof(float) * CAVE_MAX_WALL_ID);  
  for (int i=0; i<CAVE_MAX_WALL_ID; i++) {  // set the default "no one" value
    draw_flags[i] = -1;
  }
  for (i=0; i < (CAVEConfig -> ActiveWalls); i++) {  // find the ones that exist
    if (CAVEConfig -> DisplayWall[i]) {  
      draw_flags[CAVEConfig->DisplayWall[i]] = 0;
    }
  }
  
}


////////////////////////////  destructor  
CaveScene::~CaveScene(void) {
  MSGDEBUG(1,"Deleting (avoiding) a Normal Scene ..." << sendmsg);

  clear();              // unregisters all displayables
  free_to_CAVE_memory(draw_flags);  // and free shared memory
  free_to_CAVE_memory(cave_numDisplayable3D);
  free_to_CAVE_memory(cave_numDisplayable2D);
  free_to_CAVE_memory(cave_num_disp_list3D);
  free_to_CAVE_memory(cave_num_disp_list2D);
}

// called by the CAVE display function call
// for the children rendering processes!
// this sets up the correct local variables, then calls draw
void CaveScene::draw(DisplayDevice *display) {
  if (draw_flags[CAVEWall] == -1) {
    msgErr << "update proc. thought I " << CAVEWall << " don't exist";
    msgErr << sendmsg;
    exit(1);
  }
  while(draw_flags[CAVEWall] == 0)  // busy wait for the start
     ;
  numDisplayable3D = *cave_numDisplayable3D;
  numDisplayable2D = *cave_numDisplayable2D;
  numCmdLists3D = *cave_num_disp_list3D;
  numCmdLists2D = *cave_num_disp_list2D;
  
  Scene::draw(display);
  draw_flags[CAVEWall] = 0;
}

// called in VMDupdate, this updates the values of numDisplayable[23]D
// this is called by the parent!
void CaveScene::prepare(DisplayDevice *display) {

  // check if the ESC key is being pressed; if so, quit
  // note: THIS IS A HACK, just for Bill Sherman
  if(getbutton(ESCKEY)) {
    CAVEExit();
    exit(10);
  }

  Scene::prepare(display);
  *cave_numDisplayable3D = numDisplayable3D;
  *cave_numDisplayable2D = numDisplayable2D;
  *cave_num_disp_list3D = numCmdLists3D;
  *cave_num_disp_list2D = numCmdLists2D;
  
  int i;
  for (i=0; i<CAVE_MAX_WALL_ID; i++) {  // start 'em off
    if (draw_flags[i] == 0)
       draw_flags[i] = 1;
  }
  int flg = 0;
  while (!flg) {
    flg = 1;
    for (i=0; i<CAVE_MAX_WALL_ID; i++) { // wait for the finish
      if (draw_flags[i] > 0)
        flg = 0;
    }
  }
}

//////////////////////////  public virtual functions  

// return Nth 2D Displayable object; NULL if not possible
Displayable *CaveScene::displayable2D(int n) {
  if(n >= 0 && n < num_displayable2D())
    return dispObjList2D[n];
  else
    return NULL;
}
  
// return Nth 3D Displayable object; NULL if not possible
Displayable *CaveScene::displayable3D(int n) {
  if(n >= 0 && n < num_displayable3D())
    return dispObjList3D[n];
  else
    return NULL;
}
  
// return Nth 2D command list pointer; NULL if not possible
char *CaveScene::disp_list2D(int n) {
  if(n >= 0 && n < num_disp_list2D())
    return cmdListArray2D[n];
  else
    return NULL;
}
    
// return Nth 3D command list pointer; NULL if not possible
char *CaveScene::disp_list3D(int n) {
  if(n >= 0 && n < num_disp_list3D())
    return cmdListArray3D[n];
  else
    return NULL;
}


// allocate display list storage space of specified size
char *CaveScene::get_disp_storage(int sz) {
  // for this version of the scene, just malloc it
  return (char *) malloc_from_CAVE_memory(sz);
}


// free display list storage space 
void CaveScene::free_disp_storage(char *displist) {

  // for this version of the scene, just delete it
  if(displist)
    free_to_CAVE_memory(displist);
}


// register the given Displayable and corresponding display list (of dim d)
// If disp is NULL, just store the displist.
void CaveScene::Register(Displayable *disp, char *displist, int d) {

  // make sure we are not already managing this object
  if(disp) {
    int found = (disp->dim() == 3 ? 
          dispObjList3D.find(disp) : dispObjList2D.find(disp));
    if(found >= 0) {
      MSGDEBUG(1,"Scene: attempted to register Displayable '" << disp->name);
      MSGDEBUG(1,"' to scene after already registered." << sendmsg);
      return;
    } else {
      MSGDEBUG(1,"Scene: registering Displayable '" << disp->name << "' ...");
      MSGDEBUG(1,sendmsg);
    }

    // add the Displayable pointer to list
    if(disp->dim() == 3) {
      dispObjList3D.append(disp);
      numDisplayable3D = dispObjList3D.num();
    } else {
      dispObjList2D.append(disp);
      numDisplayable2D = dispObjList2D.num();
    }
    MSGDEBUG(2,"     : Now storing " << num_displayable2D() << " 2D and ");
    MSGDEBUG(2,num_displayable3D() << " 3D Displayable's." << sendmsg);  
  }
  
  if(displist) {
    MSGDEBUG(1,"Scene: registering new display list." << sendmsg);
    MSGDEBUG(3,"     : first three values in display list: ");
    MSGDEBUG(3,((int *)displist)[0] << ", " << ((int *)displist)[1] << ", ");
    MSGDEBUG(3,((int *)displist)[2]);
    MSGDEBUG(3,sendmsg);

    // add the display list to our own lists
    if(d == 3) {
      cmdListArray3D.append(displist);
      numCmdLists3D = cmdListArray3D.num();
    } else {
      cmdListArray2D.append(displist);
      numCmdLists2D = cmdListArray2D.num();
    }
    MSGDEBUG(2,"     : Now storing " << num_disp_list2D() << " 2D and ");
    MSGDEBUG(2, num_disp_list3D() << " 3D command lists." << sendmsg);
  }
}


// unregister the given Displayable and corresponding display list (of dim d).
// if disp is NULL, just unregister the given display list
void CaveScene::unRegister(Displayable *disp, char *displist, int d) {
  int found;

  // make sure we are managing this object
  if(disp) {
    found = (disp->dim() == 3 ? 
          dispObjList3D.find(disp) : dispObjList2D.find(disp));
    if(found < 0) {
      MSGDEBUG(1,"Scene: attempted to unregister Displayable '" << disp->name);
      MSGDEBUG(1,"' to scene when it is not in the scene." << sendmsg);
      return;
    }
  
    // remove the pointer from list
    if(disp->dim() == 3) {
      dispObjList3D.remove(found);
      numDisplayable3D = dispObjList3D.num();
    } else {
      dispObjList2D.remove(found);
      numDisplayable2D = dispObjList2D.num();
    }
    MSGDEBUG(1,"Scene: unregistered Displayable " << found << ":'");
    MSGDEBUG(1,disp->name << "' ..." << sendmsg);
  }
  
  if(displist) {
    found = (d == 3 ? cmdListArray3D.find(displist) :
                                 cmdListArray2D.find(displist));
    if(found < 0) {
      MSGDEBUG(1,"Scene: attempted to unregister unknown display list");
      MSGDEBUG(1," from scene when it is not in the scene." << sendmsg);
      return;
    }
      
    // remove the pointer from list
    if(d == 3) {
      cmdListArray3D.remove(found);
      numCmdLists3D = cmdListArray3D.num();
    } else {
      cmdListArray2D.remove(found);
      numCmdLists2D = cmdListArray2D.num();
    }
  
    MSGDEBUG(1,"Scene: unregistered display list " << found << sendmsg);
  }
}

