/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdDisplay.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.4 $	$Date: 1995/05/11 22:05:43 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   These are the Commands that control the various aspects
 * of the display, like, the clipping planes, eye separation, etc.
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdDisplay.h"
#include "DisplayDevice.h"
#include "Scene.h"
#include "ColorList.h"
#include "LightList.h"
#include "Axes.h"
#include "Stage.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

// These are the Command:: enums used by this file:
//  DISP_RESHAPE, DISP_STEREO, DISP_EYESEP, DISP_FOCALLEN, 
//  DISP_LIGHT_ON, DISP_LIGHT_HL, DISP_LIGHT_ROT, DISP_MATERIALS_CHANGE,
//  DISP_CLIP, DISP_DEPTHCUE, DISP_ANTIALIAS, DISP_SCRHEIGHT, DISP_SCRDIST,
//  CMD_AXES, CMD_STAGE


////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'display'; return TRUE if an error occurs.
int text_cmd_display(int argc, char **argv, CommandQueue *cmdQueue, int id) {

  if(argc == 2) {
    if(!strupncmp(argv[1],"reshape",CMDLEN))
      cmdQueue->append(new CmdDisplayReshape(id));
    else if(!strupncmp(argv[1],"resetview",CMDLEN))
      cmdQueue->append(new CmdResetView(id));
    else
      return TRUE;

  } else if(argc == 3) {
    if(!strupncmp(argv[1],"eyesep",CMDLEN)) {
      cmdQueue->append(new CmdDisplayEyesep(atof(argv[2]), id));
    } else if(!strupncmp(argv[1],"focallength",CMDLEN)) {
      cmdQueue->append(new CmdDisplayFocallen(atof(argv[2]), id));
    } else if(!strupncmp(argv[1],"height",CMDLEN)) {
      cmdQueue->append(new CmdDisplayScreenHeight(atof(argv[2]), id));
    } else if(!strupncmp(argv[1],"distance",CMDLEN)) {
      cmdQueue->append(new CmdDisplayScreenDistance(atof(argv[2]), id));
    } else if(!strupncmp(argv[1],"antialias",CMDLEN)) {
      cmdQueue->append(new CmdDisplayAAOn(!strupcmp(argv[2],"on"), id));
    } else if(!strupncmp(argv[1],"depthcue",CMDLEN)) {
      cmdQueue->append(new CmdDisplayDepthcueOn(!strupcmp(argv[2],"on"), id));
    } else if(!strupncmp(argv[1],"stereo",CMDLEN)) {
      int i, modes = display->num_stereo_modes();
      for(i=0; i < modes; i++)
        if(!strupncmp(argv[2],display->stereo_name(i), CMDLEN))
	  break;
      cmdQueue->append(new CmdDisplayStereo(i, id));
    } else
      return TRUE;

  } else if(argc == 4) {
    if(!strupncmp(argv[1],"nearclip",CMDLEN)) {
      if(!strupncmp(argv[2],"set",CMDLEN))
	cmdQueue->append(new CmdDisplayClipNear(atof(argv[3]), id));
      else if(!strupncmp(argv[2],"add",CMDLEN))
	cmdQueue->append(new CmdDisplayClipNearRel(atof(argv[3]), id));
      else
        return TRUE;
    } else if(!strupncmp(argv[1],"farclip",CMDLEN)) {
      if(!strupncmp(argv[2],"set",CMDLEN))
	cmdQueue->append(new CmdDisplayClipFar(atof(argv[3]), id));
      else if(!strupncmp(argv[2],"add",CMDLEN))
	cmdQueue->append(new CmdDisplayClipFarRel(atof(argv[3]), id));
      else
        return TRUE;
    } else
      return TRUE;
  }

  // if here, completed successfully
  return FALSE;
}


// text callback routine for 'light'; return TRUE if an error occurs.
int text_cmd_light(int argc, char **argv, CommandQueue *cmdQueue, int id) {

  if(argc > 1) {
    int n = atoi(argv[1]);
    if(argc == 3) {
      if(!strupncmp(argv[2],"on",CMDLEN))
	cmdQueue->append(new CmdDisplayLightOn(n, TRUE, id));
      else if(!strupncmp(argv[2],"off",CMDLEN))
	cmdQueue->append(new CmdDisplayLightOn(n, FALSE, id));
      else if(!strupncmp(argv[2],"highlight",CMDLEN))
	cmdQueue->append(new CmdDisplayLightHL(n, TRUE, id));
      else if(!strupncmp(argv[2],"unhighlight",CMDLEN))
	cmdQueue->append(new CmdDisplayLightHL(n, FALSE, id));
      else
	return TRUE;

    } else if(argc == 5 && !strupncmp(argv[2],"rot",CMDLEN)) {
      char axis = (char)(tolower(argv[3][0]));
      float deg = atof(argv[4]);
      cmdQueue->append(new CmdDisplayLightRot(n, deg, axis, id));

    } else
      return TRUE;

  } else
    return TRUE;

  // if here, completed successfully
  return FALSE;
}


// text callback routine for 'axes'; return TRUE if an error occurs.
int text_cmd_axes(int argc, char **argv, CommandQueue *cmdQueue, int id) {
  if(axes && argc == 3) {
    if(!strupncmp(argv[1],"location",CMDLEN)) {
      for(int i=0; i < axes->locations(); i++)
	if(!strupncmp(argv[2],axes->loc_description(i),CMDLEN))
	  break;
      cmdQueue->append(new CmdDisplayAxes(i, id));
    } else
      return TRUE;
  } else
    return TRUE;
 
  // if here, completed successfully
  return FALSE;
}


// text callback routine for 'stage'; return TRUE if an error occurs.
int text_cmd_stage(int argc, char **argv, CommandQueue *cmdQueue, int id) {
  if(stage && argc == 3) {
    if(!strupncmp(argv[1],"location",CMDLEN)) {
      for(int i=0; i < stage->locations(); i++)
	if(!strupncmp(argv[2],stage->loc_description(i),CMDLEN))
	  break;
      	cmdQueue->append(new CmdDisplayStage(CmdDisplayStage::LOCATION,
				i, id));
    } else if(!strupncmp(argv[1],"panels",CMDLEN)) {
      cmdQueue->append(new CmdDisplayStage(CmdDisplayStage::PANELS,
				atoi(argv[2]), id));
    } else
      return TRUE;

  } else
    return TRUE;
 
  // if here, completed successfully
  return FALSE;
}


//////////////////// reshape the display
int CmdDisplayReshape::do_execute(void) {
  int retval;
  if(retval = (display != NULL))
    display->reshape();
  return retval;
}

CmdDisplayReshape::CmdDisplayReshape(int newUIid)
  : Command(Command::DISP_RESHAPE, newUIid) { }


///////////////// reset the current view for the current scene
int CmdResetView::do_execute(void) {
  int retval;
  if(retval = (scene != NULL))
    scene->reset_transformation();
  return retval;
}

void CmdResetView::create_text(void) {
  *cmdText << "display resetview" << ends;
}

CmdResetView::CmdResetView(int newUIid)
  : Command(Command::DISP_RESETVIEW, newUIid) {
}

//////////////////// set stereo mode of display
int CmdDisplayStereo::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    if(retval = (stmode >= 0 && stmode < display->num_stereo_modes()))
      display->set_stereo_mode(stmode);
    else
      msgErr << "Cannot set stereo: unknown stereo mode specified." <<sendmsg;
  }
  return retval;
}

void CmdDisplayStereo::create_text(void) {
  if(display && stmode >= 0 && stmode < display->num_stereo_modes())
    *cmdText << "display stereo " << display->stereo_name(stmode);
  else
    *cmdText << "display stereo " << stmode;
  *cmdText << ends;
}

CmdDisplayStereo::CmdDisplayStereo(int newstmode, int newUIid)
  : Command(Command::DISP_STEREO, newUIid) {
  stmode = newstmode;
}


//////////////////// set eye separation of display
int CmdDisplayEyesep::do_execute(void) {
  int retval;
  if(retval = (display != NULL))
    display->set_eyesep(sep);
  return retval;
}

void CmdDisplayEyesep::create_text(void) {
  *cmdText << "display eyesep " << sep << ends;
}

CmdDisplayEyesep::CmdDisplayEyesep(float newsep, int newUIid)
  : Command(Command::DISP_EYESEP, newUIid) {
  sep = newsep;
}


//////////////////// set focal length of display
int CmdDisplayFocallen::do_execute(void) {
  int retval;
  if(retval = (display != NULL
      && flen != 0.0 && display->eye_dist() != 0.0)) {
    float lx, ly, lz, fl;
    display->where_looking(lx, ly, lz);
    fl = flen / (display->eye_dist());
    lx *= fl;  ly *= fl;  lz *= fl;
    display->change_look(lx, ly, lz);
  }
  return retval;
}

void CmdDisplayFocallen::create_text(void) {
  *cmdText << "display focallength " << flen << ends;
}

CmdDisplayFocallen::CmdDisplayFocallen(float newlen, int newUIid)
  : Command(Command::DISP_FOCALLEN, newUIid) {
  flen = newlen;
}


//////////////////// set screen height value
int CmdDisplayScreenHeight::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    display->screen_height(val);
  }
  return retval;
}

void CmdDisplayScreenHeight::create_text(void) {
  *cmdText << "display height " << val << ends;
}

CmdDisplayScreenHeight::CmdDisplayScreenHeight(float newval, int newUIid)
  : Command(Command::DISP_SCRHEIGHT, newUIid) {
  val = newval;
}


//////////////////// set distance to screen from origin
int CmdDisplayScreenDistance::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    display->distance_to_screen(val);
  }
  return retval;
}

void CmdDisplayScreenDistance::create_text(void) {
  *cmdText << "display distance " << val << ends;
}

CmdDisplayScreenDistance::CmdDisplayScreenDistance(float newval, int newUIid)
  : Command(Command::DISP_SCRDIST, newUIid) {
  val = newval;
}


//////////////////// turn on/off antialiasing
int CmdDisplayAAOn::do_execute(void) {
  int retval;
  if(retval = (display != NULL && display->aa_available())) {
    if(onoff)
      display->aa_on();
    else
      display->aa_off();
  }
  return retval;
}

void CmdDisplayAAOn::create_text(void) {
  *cmdText << "display antialias " << (onoff ? "on" : "off") << ends;
}

CmdDisplayAAOn::CmdDisplayAAOn(int turnon, int newUIid)
  : Command(Command::DISP_ANTIALIAS, newUIid) {
  onoff = turnon;
}


//////////////////// turn on/off depth cueing
int CmdDisplayDepthcueOn::do_execute(void) {
  int retval;
  if(retval = (display != NULL && display->cueing_available())) {
    if(onoff)
      display->cueing_on();
    else
      display->cueing_off();
  }
  return retval;
}

void CmdDisplayDepthcueOn::create_text(void) {
  *cmdText << "display depthcue " << (onoff ? "on" : "off") << ends;
}

CmdDisplayDepthcueOn::CmdDisplayDepthcueOn(int turnon, int newUIid)
  : Command(Command::DISP_DEPTHCUE, newUIid) {
  onoff = turnon;
}


/////////////////////  clipping plane controls
// This handles the whole range of clipping plane options
//  There are derived classes so you won't have to have the funky flags
// to change {near,fixed} clipping plane {to a fixed,by a relative} amount
// or not
int CmdDisplayClip::do_execute(void) {
  int retval;
  if(retval = (display != NULL)) {
    if (changenear) {
      if (setval)
        display -> set_near_clip(amount);
      else
        display->addto_near_clip(amount);
    } else {
      if (setval)
        display -> set_far_clip(amount);
      else
        display->addto_far_clip(amount);
    }
  }
  return retval;
}

void CmdDisplayClip::create_text(void) {
  *cmdText << "display " << (changenear ? "near" : "far");
  *cmdText << "clip " << (setval ? "set " : "add ");
  *cmdText << amount << ends;
}

CmdDisplayClip::CmdDisplayClip(int ischangenear, int issetval, 
                                float newamt, int UIid)
  : Command(Command::DISP_CLIP, UIid) {
  changenear = ischangenear;
  setval = issetval;
  amount = newamt;
}  

 
////////////////////  change the axes location
int CmdDisplayAxes::do_execute(void) {
  int retval;
  if(retval=(axes != NULL)) {
    if(retval = (newpos >= 0 && newpos < axes->locations()))
      retval = axes->location(newpos);
    else
      msgErr << "Cannot change axes location: unknown location." << sendmsg;
  }
  return retval;
}

void CmdDisplayAxes::create_text(void) {
  if((axes != NULL) && newpos >= 0 && newpos < axes->locations())
    *cmdText << "axes location " << axes->loc_description(newpos);
  *cmdText << ends;
}

CmdDisplayAxes::CmdDisplayAxes(int npos, int newUIid)
  : Command(Command::CMD_AXES, newUIid) {
  newpos = npos;
}
    

////////////////////  change the stage location
int CmdDisplayStage::do_execute(void) {
  int retval;
  if(settingToChange == LOCATION) {
    if(retval=(stage !=NULL)) {
      if(retval = (newpos >= 0 && newpos < stage->locations()))
        retval = stage->location(newpos);
      else
        msgErr << "Cannot change stage location: unknown location." <<sendmsg;
    }
  } else if(settingToChange == PANELS) {
    if(retval=(stage !=NULL))
      retval = stage->panels(newpos);
  } else
    retval = FALSE;
  return retval;
}

void CmdDisplayStage::create_text(void) {
  if((stage != NULL)) {
    *cmdText << "stage ";
    if(settingToChange == LOCATION && newpos >= 0 && 
        newpos < stage->locations()) {
      *cmdText << "location " << stage->loc_description(newpos);
    } else if(settingToChange == PANELS) {
      *cmdText << "panels " << newpos;
    }
  }
  *cmdText << ends;
}

CmdDisplayStage::CmdDisplayStage(Settings toset, int npos, int newUIid)
  : Command(Command::CMD_STAGE, newUIid) {
  newpos = npos;
  settingToChange = toset;
}


//////////////////// turn on/off the Nth light
int CmdDisplayLightOn::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    if(onoff)
      (lights->current())->on();
    else
      (lights->current())->off();
  }
  return retval;
}

void CmdDisplayLightOn::create_text(void) {
  *cmdText << "light " << n << (onoff ? " on" : " off") << ends;
}

CmdDisplayLightOn::CmdDisplayLightOn(int ln, int turnon, int newUIid)
  : Command(Command::DISP_LIGHT_ON, newUIid) {
  n = ln;  onoff = turnon;;
}


//////////////////// highlight the Nth light
int CmdDisplayLightHL::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    (lights->current())->highlight(hl);
  }
  return retval;
}

void CmdDisplayLightHL::create_text(void) {
  *cmdText << "light " << n << (hl ? " highlight" : " unhighlight") << ends;
}

CmdDisplayLightHL::CmdDisplayLightHL(int ln, int highlt, int newUIid)
  : Command(Command::DISP_LIGHT_HL, newUIid) {
  n = ln;  hl = highlt;
}


//////////////////// rotate the position of the Nth light
int CmdDisplayLightRot::do_execute(void) {
  int retval;
  if(retval = (lights != NULL && n >= 0 && n < DISP_LIGHTS)) {
    lights->set(n);
    (lights->current())->add_rotation(theta, axis);
  }
  return retval;
}

void CmdDisplayLightRot::create_text(void) {
  *cmdText << "light " << n << " rot " << axis << " " << theta << ends;
}

CmdDisplayLightRot::CmdDisplayLightRot(int ln, float th, char ax,
  int newUIid): Command(Command::DISP_LIGHT_ROT, newUIid) {
  n = ln;  theta = th;  axis = ax;
}


