/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdMol.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.16 $	$Date: 1995/05/23 03:21:44 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   Command objects for affecting molecules.
 *
 ***************************************************************************/

#include <ctype.h>
#include <stdlib.h>
#include "CmdMol.h"
#include "MoleculeFilePDB.h"
#include "MoleculeFilePSF.h"
#include "MoleculeFileEDM.h"
#include "MoleculeFileGraph.h"
#include "MoleculeFileRaster3D.h"
#include "BabelConvert.h"
#include "UIText.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"
  
// except this, for remote connections
#ifdef VMDREMOTE
#include "MoleculeRemote.h"
#include "RemoteList.h"
#endif

// include info about Sigma molecule, if enabled
#ifdef VMDSIGMA
#include "CmdSigma.h"
#include "MoleculeSigma.h"
#endif

// the following uses the Cmdtypes MOL_NEW, MOL_LIST, MOL_DEL, MOL_ACTIVE,
// MOL_FIX, MOL_ON, MOL_TOP, MOL_SELECT, MOL_REP, MOL_COLOR, MOL_ADDREP,
// MOL_MODREP, MOL_DELREP, MOL_MODREPITEM


/*
 * NOTES:
 *
 *	1) When referring to a molecule in a command by a number, the
 * unique molecule ID is used, NOT the relative index of the molecule into
 * the molecule list.  This is because the relative index changes as molecules
 * are added/removed, but the unique ID stays the same until the molecule is
 * deleted.
 *
 *	2) All commands which have a particular molecule being referred to
 * take a string argument, which can be "all", "top", "active", "displayed",
 * "none", or a number which is a molecule ID as explained in note #1.
 *
 *	3) Almost all Molecule commands are derived from the general base
 * class "CmdMolecule" (except for commands which do not do any action to an
 * existing molecule, such as CmdMolNew)
 * which stores and frees the string used to refer to the molecule.
 *
 */

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'mol'; return TRUE if an error occurs.
int text_cmd_mol(int argc, char **argv, CommandQueue *cmdQueue, int id) {

  if((argc == 4 || argc == 6) && (!strupncmp(argv[1], "load", CMDLEN) ||
		!strupncmp(argv[1], "new", CMDLEN)) ) {
    // load a new molecule
    int sftype = MoleculeFile::UNKNOWN;
    int cftype = CoorFileData::UNKNOWN;
    char *cfstr = NULL;
    int i;

    // find structure file type
    for(i=0; i < MoleculeFile::STRTYPES; i++) {
      if(!strupncmp(argv[2], structureFileTypeNames[i], CMDLEN))
	sftype = i;
    }
    if(sftype == MoleculeFile::UNKNOWN) {
      msgErr << "Unknown structure file type " << argv[2]  << sendmsg;
    }

    if(argc == 6) {
      // find coordinate file type
      for(i=0; i < CoorFileData::COORTYPES; i++) {
        if(!strupncmp(argv[4], CoorFileSuffix[i], CMDLEN))
	  cftype = i;
      }
      if(cftype == CoorFileData::UNKNOWN) {
	msgErr << "Unknown coordinate file type " << argv[4] << sendmsg;
      } else {
	cfstr = argv[5];
      }
    }
	
    if(sftype != MoleculeFile::UNKNOWN) {
#ifdef VMDSIGMA
      extern int have_sigma;    // In startup.C
      if (have_sigma)
        cmdQueue->append(new CmdMolNew(argv[3],sftype,cfstr,cftype,
                                        CmdMolNew::MOLSIGMA, id));
      else
#endif
      cmdQueue->append(new CmdMolNew(argv[3],sftype,cfstr,cftype,id));
    } else
      return TRUE;		// signal an error

  } else if(argc < 4 && !strupncmp(argv[1], "list", CMDLEN)) {
    if(argc == 2)
      cmdQueue->append(new CmdMolList("all", id));
    else
      cmdQueue->append(new CmdMolList(argv[2] ,id));

  } else if(!strupncmp(argv[1],"selection",CMDLEN)) {
    char *molstr = combine_arguments(argc, argv, 2);
    cmdQueue->append(new CmdMolSelect(molstr, id));
    if(molstr)
      delete [] molstr;

  } else if(!strupncmp(argv[1],"representation",CMDLEN) ||
  	!strupncmp(argv[1],"rep",CMDLEN) ) {
    char *molstr = combine_arguments(argc, argv, 2);
    cmdQueue->append(new CmdMolRep(molstr, id));
    if(molstr)
      delete [] molstr;

  } else if(!strupncmp(argv[1],"color",CMDLEN)) {
    char *molstr = combine_arguments(argc, argv, 2);
    cmdQueue->append(new CmdMolColor(molstr, id));
    if(molstr)
      delete [] molstr;

  } else if(argc > 4 && (!strupncmp(argv[1],"modcolor",CMDLEN) ||
  	!strupncmp(argv[1],"modstyle",CMDLEN) ||
	!strupncmp(argv[1],"modselect",CMDLEN)) ) {
    char *molstr = combine_arguments(argc, argv, 4);
    if(!strupncmp(argv[1],"modcolor",CMDLEN))
      cmdQueue->append(new CmdMolChangeRepItem(atoi(argv[2]), argv[3],
      		 CmdMolChangeRepItem::COLOR, molstr, id));
    else if(!strupncmp(argv[1],"modstyle",CMDLEN))
      cmdQueue->append(new CmdMolChangeRepItem(atoi(argv[2]), argv[3],
      		 CmdMolChangeRepItem::REP, molstr, id));
    else if(!strupncmp(argv[1],"modselect",CMDLEN))
      cmdQueue->append(new CmdMolChangeRepItem(atoi(argv[2]), argv[3],
      		 CmdMolChangeRepItem::SEL, molstr, id));
    if(molstr)
      delete [] molstr;

  } else if(argc < 4 && argc > 1 && !strupncmp(argv[1],"addrep",CMDLEN)) {
    cmdQueue->append(new CmdMolAddRep("top", id));

  } else if(argc == 4 && !strupncmp(argv[1],"delrep",CMDLEN)) {
    cmdQueue->append(new CmdMolDeleteRep(atoi(argv[2]), argv[3], id));

  } else if(argc == 4 && !strupncmp(argv[1],"modrep",CMDLEN)) {
    cmdQueue->append(new CmdMolChangeRep(atoi(argv[2]), argv[3], id));

  } else if(argc == 3 && !strupncmp(argv[1],"delete",CMDLEN)) {
    cmdQueue->append(new CmdMolDelete(argv[2], id));

  } else if(argc == 3 && (!strupncmp(argv[1],"active",CMDLEN) ||
  			!strupncmp(argv[1],"inactive",CMDLEN))) {
    cmdQueue->append(new CmdMolActive(argv[2],
    	(strupncmp(argv[1],"active",CMDLEN) == 0), id));

  } else if(argc == 3 && (!strupncmp(argv[1],"on",CMDLEN) ||
  			!strupncmp(argv[1],"off",CMDLEN))) {
    cmdQueue->append(new CmdMolOn(argv[2],
    	(strupncmp(argv[1],"on",CMDLEN) == 0), id));

  } else if(argc == 3 && (!strupncmp(argv[1],"fix",CMDLEN) ||
  			!strupncmp(argv[1],"free",CMDLEN))) {
    cmdQueue->append(new CmdMolFix(argv[2],
    	(strupncmp(argv[1],"fix",CMDLEN) == 0), id));

  } else if(argc == 3 && !strupncmp(argv[1],"top",CMDLEN)) {
    cmdQueue->append(new CmdMolTop(argv[2], id));

  } else
    return TRUE;
  
  return FALSE;
}


////////////////////////////////  CmdMolecule: base class  ////////////////

// calculate which molecules to operate upon from the given molecule name;
// put the data in idList, and return the pointer to the integer list.
// If NULL is returned, there was an error or no molecules were specified.
// If an error occurs, this prints out the error message as well.
//
// Molecule names are of the form
//		<n1>[|<n2>[.... [|<nN>]...]]
// where <ni> is either "all", "top", "active", "inactive", "displayed",
// "on", "off", "fixed", "free", "none", or an ID.
// There should be no spaces in the name.
int *CmdMolecule::find_mol_list(char *givenName) {
  char name[512];		// max size the name can be is 512 bytes.
  int numMols;
  register char *tok;
  register int i, n;

  // make sure we have a molecule and a name to check
  if(!givenName)
    return NULL;

  if(! moleculeList->num() ) {
    msgErr << "No molecules are currently loaded." << sendmsg;
    return NULL;
  }

  // if so, allocate enough space to store all the current molecules, and use
  // it as an array of flags for the present
  i = numMols = moleculeList->num();
  if(idList)
    delete [] idList;
  idList = new int[i];
  numMol = 0;
  while(--i >= 0)
    idList[i] = FALSE;

  // start to tokenize the string, then
  // scan the string and look for molecule specifiers
  strcpy(name, givenName);
  tok = strtok(name, "|/:");
  while(tok) {
    // look for what kind of name this is
    if(isdigit(*tok)) {		// check if it is a number
      n = atoi(tok);
      i = moleculeList->mol_index_from_id(n);
      if(i < 0) {
        // bad ID; print error and exit
        molname_error(givenName, tok);
        return NULL;
      }
      idList[i] = TRUE;
    } else if(!strupncmp(tok, "all", CMDLEN)) {		// check if "all"
      for(i=0; i < numMols; i++)
        idList[i] = TRUE;
    } else if(!strupncmp(tok, "none", CMDLEN)) {	// "none"
      for(i=0; i < numMols; i++)
        idList[i] = FALSE;
    } else if(!strupncmp(tok, "top", CMDLEN)) {		// "top"
      for(i=0; i < numMols; i++)
        idList[i] = idList[i] || moleculeList->is_top(i);
    } else if(!strupncmp(tok, "active", CMDLEN)) {	// check if "active"
      for(i=0; i < numMols; i++)
        idList[i] = idList[i] || moleculeList->active(i);
    } else if(!strupncmp(tok, "inactive", CMDLEN)) {	// check if "inactive"
      for(i=0; i < numMols; i++)
        idList[i] = idList[i] || (! moleculeList->active(i) );
    } else if(!strupncmp(tok, "displayed", CMDLEN) ||
	!strupncmp(tok, "on", CMDLEN)) { 		// "displayed" or "on"
      for(i=0; i < numMols; i++)
        idList[i] = idList[i] || moleculeList->displayed(i);
    } else if(!strupncmp(tok, "off", CMDLEN)) {	// check if "off"
      for(i=0; i < numMols; i++)
        idList[i] = idList[i] || (! moleculeList->displayed(i) );
    } else if(!strupncmp(tok, "fixed", CMDLEN)) {	// check if "fixed"
      for(i=0; i < numMols; i++)
        idList[i] = idList[i] || moleculeList->fixed(i);
    } else if(!strupncmp(tok, "free", CMDLEN) ||
	!strupncmp(tok, "unfix", CMDLEN)) { 		// "free" or "unfix"
      for(i=0; i < numMols; i++)
        idList[i] = idList[i] || (! moleculeList->fixed(i) );
    } else {
      // bad molecule name; print error and exit
      molname_error(givenName, tok);
      return NULL;
    }
    tok = strtok(NULL,"|/:");
  }

  // found the names; now convert the flag array to a list of indices.  Note
  // that these are the indices, NOT the ID's as they were given by the user.
  for(i=0; i < numMols; i++) {
    if(idList[i])
      idList[numMol++] = i;
  }

  // return the information
  return (numMol > 0 ? idList : NULL);
}


// display an error message that the molecule specification is in error.
// arguments are total name, and part that had problems
void CmdMolecule::molname_error(char *fullname, char *errpart) {
  msgErr << "Illegal molecule specification '" << fullname << "': Could not ";
  msgErr << "find molecule '" << errpart << "'." << sendmsg;
}


// constructor: save name, init variables
CmdMolecule::CmdMolecule(char *name, Cmdtype code, int newUIid)
	: Command(code, newUIid) {

  // save name and init data
  whichMol = stringdup(name);
  idList = NULL;
  numMol = 0;
}

// destructor: free up storage
CmdMolecule::~CmdMolecule(void) {
  delete [] whichMol;
  if(idList)
    delete [] idList;
}


///////// load a new molecule, either from a file, or from a remote connection
 int CmdMolNew::do_execute(void) {
    Molecule *newmol = NULL;
    int babel_num = -1;
    if(moleculeList != NULL) {
    
      // first get the proper type of molecule object ready
      if(molSource == MOLFILE) {
        // read in a new molecule from a file
	 switch(molStrFileType) {
	  case MoleculeFile::PDB :
	    newmol = new MoleculeFilePDB(molStrFile, moleculeList);
	    break;
	  case MoleculeFile::PSF :
	    newmol = new MoleculeFilePSF(molStrFile, moleculeList);
	    break;
	  case MoleculeFile::EDM :
	    newmol = new MoleculeFileEDM(molStrFile, moleculeList);
	    break;
	  case MoleculeFile::GRAPH :
	    newmol = new MoleculeFileGraph(molStrFile, moleculeList);
	    break;
	  case MoleculeFile::RASTER3D :
	    newmol = new MoleculeFileRaster3D(molStrFile, moleculeList);
	    break;
	  default:  // easier than listing about 45 file types
	    {
	       // do the conversion
	       babel_num = babel_convert(molStrFileType, molStrFile);
	       if (babel_num == 0) {   // nothing happened
		  msgErr << "Unable to use Babel to convert to PDB format.  ";
		  msgErr << "Check that Babel is installed on your system,\n";
		  msgErr << "and that VMDBABELBIN is set to the location of";
		  msgErr << " the Babel executable." << sendmsg;
		  newmol = NULL;
		  break;
	       }
	       // get filename
	       const char *s = babel_file(molStrFile, 0, babel_num > 1);
	       // and set it (the molStrFile is the name in the mol window)
	       newmol = new MoleculeFilePDB((char *)s, molStrFile,
						moleculeList);
	       if (!newmol) {                  // if it didn't work,
		  babel_delete_all(molStrFile);// delete the converted files
		  break;
	       }
	    }
	    break;
	  case MoleculeFile::STRTYPES:
	  case MoleculeFile::UNKNOWN:
	    msgErr << "Cannot understand that file type\n" << sendmsg;
	    return FALSE;
	 }
      } else if(molSource == MOLREMOTE) {
#ifdef VMDREMOTE
        // read in molecule from a remote connection
	MoleculeRemote *remmol = new MoleculeRemote(remote, moleculeList);
	newmol = remmol;
	remote = NULL;
	
	// add the remote molecule to the remoteList
	if(remoteList)
	  remoteList->add_molecule(remmol);
#else
        msgErr<<"Remote connection capability has not been included in this";
	msgErr<<"executable.\nRecompile with VMDREMOTE defined." << sendmsg;
	return(FALSE);
#endif
      } else if(molSource == MOLSIGMA) {
#ifdef VMDSIGMA
        if (molStrFileType != MoleculeFile::PDB) {
            msgErr << "CmdMolNew: SIgMA requires a PDB file" << sendmsg;
            return(FALSE);
        }
        msgInfo << "CmdMolNew::do_execute: creating MoleculeSigma" << sendmsg;
        newmol = new MoleculeSigma(molStrFile, moleculeList);
#else
        msgErr << "SIgMA connection capability has not been included in this";
        msgErr << "executable.\nRecompile with VMDSIGMA defined." << sendmsg;
        return(FALSE);
#endif
      }

      // now load structure
      if(newmol)
        msgInfo << "Loading new molecule ..." << sendmsg;

      if (!newmol) {
	 msgErr << "Cannot create new molecule" << sendmsg;
	 delete newmol;
      } else if(!newmol->create()){// does real initialization and creation
	 if (babel_num > 0) {  // then I have to delete the temp babel files
	    babel_delete_all(molStrFile);
	 }
	 msgErr << "Cannot create new molecule." << sendmsg;
	 delete newmol;
	 newmol = NULL;
      } else {
        // load was successful ...
	 // now load coordinate file, if any
	 if(molCoorFile) {
	    newmol->read_coor_file(molCoorFile, molCoorFileType);
	 }
	 // add the molecule to the molecule list
	 newID = moleculeList->mol_index_from_id(
			 moleculeList->add_molecule(newmol));

	 if (babel_num > 0) {  // if this went through babel, delete the
	    unlink(babel_file(molStrFile, 0, babel_num > 1)); // first frame
	 }
	 // if there are multiple frames, queue the read/delete for the
	 // rest of the trajectory
	 if (babel_num > 1) {
	    char *t = new char[strlen(babel_file(molStrFile,1,babel_num > 1))
			       + 20 +
		       strlen("anim readdel pdb beg -1 end -1 skip 1 ")];
	    for (int i=1; i<babel_num; i++) {
	       sprintf(t, "anim readdel pdb %s beg -1 end -1 skip 1 %d",
		       babel_file(molStrFile, i, babel_num > 1),
		       newmol->id());
	       commandQueue -> append(new TextEvent(stringdup(t), TRUE));
	    }
	    delete [] t;
	 }
      }
   } // moleculeList != NULL
   return (newmol != NULL);
}

void CmdMolNew::create_text(void) {
  if(molSource == MOLFILE || molSource == MOLSIGMA) {
    *cmdText << "mol new ";
    *cmdText << structureFileTypeNames[molStrFileType] << " " << molStrFile;
    if(molCoorFile)
      *cmdText <<" "<< CoorFileSuffix[molCoorFileType] << " " << molCoorFile;
  }
  *cmdText << ends;
}

// Common initialization for constructors A and C, which read from a file
void CmdMolNew::file_init(char *fname, int sftype, char *cfile, int cftype) {
  newID = (-1);
  molStrFile = stringdup(fname);
  molStrFileType = sftype;

  if(cfile) {
    molCoorFile = stringdup(cfile);
    molCoorFileType = cftype;
  } else
    molCoorFile = NULL;
}


// constructor A: read struct from file
CmdMolNew::CmdMolNew(char *fname, int sftype, char *cfile, int cftype, 
                      int newUIid)
  : Command(Command::MOL_NEW, newUIid) {
  file_init(fname, sftype, cfile, cftype);
  molSource = MOLFILE;
}

// constructor B: read struct from remote.  Don't specify anything.
CmdMolNew::CmdMolNew(int newUIid) : Command(Command::MOL_NEW, newUIid) {
  newID = (-1);
  molSource = MOLREMOTE;
  molStrFile = molCoorFile = NULL;
}

// constructor C: read struct from file, establish link to SIgMA
CmdMolNew::CmdMolNew(char *fname, int sftype, char *cfile, int cftype,
                        MolSource /*sigma*/, int newUIid)
  : Command(Command::MOL_NEW, newUIid) {
  msgInfo << "CmdMolNew: creating MoleculeSigma for " << fname << sendmsg;
  file_init(fname, sftype, cfile, cftype);
  molSource = MOLSIGMA;
}

// destructor
CmdMolNew::~CmdMolNew(void) {
  if(molStrFile)  delete [] molStrFile;
  if(molCoorFile) delete [] molCoorFile;
}


///////////////  list the current set of loaded molecules
  // print out molecule summary line
  void CmdMolList::print_mol_summary(int i) {
    if(i >= 0 && i < moleculeList->num()) {
      Molecule *mol = moleculeList->molecule(i);
      msgInfo << mol->name;
      msgInfo << "  Atoms:" << mol->nAtoms;
      msgInfo << "  Frames (C):" << mol->num() << "(" << mol->frame() << ")";
      msgInfo << "  Status:";
      msgInfo << (moleculeList->active(mol) 	? 'A' : 'a');
      msgInfo << (moleculeList->displayed(mol)	? 'D' : 'd');
      msgInfo << (moleculeList->fixed(mol)	? 'F' : 'f');
      msgInfo << (moleculeList->is_top(mol)	? 'T' : 't');
      msgInfo << sendmsg;
    } else {
      msgErr << "Illegal molecule ID." << sendmsg;
    }
  }
  
  // print out atom rep summary line
  void CmdMolList::print_arep_summary(Molecule *mol, int i) {
    DrawMolItem *arep = mol->component(i);
    msgInfo << i << ": " << (arep->displayed() ? " on" : "off") << ", ";
    msgInfo << (arep->atomSel)->selected << " atoms selected." << sendmsg;
    msgInfo << "  Coloring method: " << (arep->atomColor)->cmdStr << sendmsg;
    msgInfo << "   Representation: " << (arep->atomRep)->cmdStr << sendmsg;
    msgInfo << "        Selection: " << (arep->atomSel)->cmdStr << sendmsg;
  }
  
int CmdMolList::do_execute(void) {
    int i,retval;
    Molecule *mol;

    if(retval = (moleculeList != NULL)) {
      if(find_mol_list(whichMol)) {
        if(numMol > 1) {
          msgInfo << "Molecule Status Overview:" << sendmsg;
          msgInfo << "-------------------------" << sendmsg;
	  for(i=0; i < numMol; i++)
	    print_mol_summary(idList[i]);
	} else if(numMol == 1) {
	  mol = moleculeList->molecule(idList[0]);
	  msgInfo << "Status of molecule " << mol->name << ":" << sendmsg;
	  print_mol_summary(idList[0]);
	  msgInfo << "Atom representations: " << mol->components() << sendmsg;
	  msgInfo << "-------------------------" << sendmsg;
	  for(i=0; i < mol->components(); i++)
	    print_arep_summary(mol, i); 
	} else {
	  retval = FALSE;
	}
      } else {
        return FALSE;
      }
    }
    return retval;
  }

void CmdMolList::create_text(void) {
  *cmdText << "mol list " << whichMol << ends;
}

CmdMolList::CmdMolList(char *wm, int newUIid)
	: CmdMolecule(wm, Command::MOL_LIST, newUIid) { }
  

///////////////  delete the Nth molecule in the molecule List
// peform actual action for molecule i.  return success.
int CmdMolDelete::do_action(int i) {
  int retval = FALSE;
  
#ifdef VMDREMOTE
  // remove from remoteList, if necessary
  remoteList->del_molecule(remoteList->rem_index_from_id(i));
#endif

  if(!moleculeList->del_molecule(moleculeList->mol_index_from_id(i)))
    msgErr << "Cannot delete molecule with ID " << i << sendmsg;
  else
    retval = TRUE;
  return retval;
}

int CmdMolDelete::do_execute(void) {
  int retval;
  int deleted = 0;

  if(retval = (moleculeList != NULL)) {
    if(find_mol_list(whichMol)) {
      // must work backwards, since deleting changes the indices
      for(int i=(numMol-1); i >= 0; i--) {
	retval = retval &&
			do_action((moleculeList->molecule(idList[i]))->id());
        if(retval)
          deleted ++;
      }
      msgInfo << "Deleted " << deleted << " molecules." << sendmsg;
    } else {
      return FALSE;
    }
  }
  return retval;
}

void CmdMolDelete::create_text(void) {
  *cmdText << "mol delete " << whichMol << ends;
}

CmdMolDelete::CmdMolDelete(char *wm, int newUIid) 
	: CmdMolecule(wm, Command::MOL_DEL, newUIid) { }


///////////////  make the Nth molecule 'active' or 'inactive'
  // peform actual action for molecule i.  return success.
  int CmdMolActive::do_action(int i) {
    int retval = FALSE;
    int m = moleculeList->mol_index_from_id(i);
    if(m >= 0) {
      retval = TRUE;
      if(yn)
        moleculeList->activate(m);
      else
        moleculeList->inactivate(m);
    } else
      msgErr << "Cannot set active status of molecule with ID "<< i <<sendmsg;
    return retval;
  }

int CmdMolActive::do_execute(void) {
  int retval;
  int activated = 0;

  if(retval = (moleculeList != NULL)) {
    if(find_mol_list(whichMol)) {
      // working backwards just for the heck of it
      for(int i=(numMol-1); i >= 0; i--) {
	retval = retval &&
			do_action((moleculeList->molecule(idList[i]))->id());
        if(retval)
          activated ++;
      }
      msgInfo << "Changed " << activated << " molecules." << sendmsg;
    } else {
      return FALSE;
    }
  }
  return retval;
}

void CmdMolActive::create_text(void) {
  *cmdText << "mol " << (yn ? "active " : "inactive ") << whichMol << ends;
}

CmdMolActive::CmdMolActive(char *wm, int newyn, int newUIid)
	: CmdMolecule(wm, Command::MOL_ACTIVE, newUIid) {
  yn = newyn;
}


///////////////  make the Nth molecule 'fixed' or 'free'
  // peform actual action for molecule i.  return success.
  int CmdMolFix::do_action(int i) {
    int retval = FALSE;
    int m = moleculeList->mol_index_from_id(i);
    if(m >= 0) {
      retval = TRUE;
      if(yn)
        moleculeList->fix(m);
      else
        moleculeList->unfix(m);
    } else
      msgErr << "Cannot set fixed status of molecule with ID " << i <<sendmsg;
    return retval;
  }

int CmdMolFix::do_execute(void) {
  int retval;
  int fixed = 0;

  if(retval = (moleculeList != NULL)) {
    if(find_mol_list(whichMol)) {
      // working backwards just for the heck of it
      for(int i=(numMol-1); i >= 0; i--) {
	retval = retval &&
			do_action((moleculeList->molecule(idList[i]))->id());
        if(retval)
          fixed ++;
      }
      msgInfo << "Changed " << fixed << " molecules." << sendmsg;
    } else {
      return FALSE;
    }
  }
  return retval;
}
  
void CmdMolFix::create_text(void) {
  *cmdText << "mol " << (yn ? "fixed " : "free ") << whichMol << ends;
}

CmdMolFix::CmdMolFix(char *wm, int newyn, int newUIid)
	: CmdMolecule(wm, Command::MOL_FIX, newUIid) {
  yn = newyn;
}


///////////////  make the Nth molecule 'on' or 'off'
  // peform actual action for molecule i.  return success.
  int CmdMolOn::do_action(int i) {
    int retval = FALSE;
    int m = moleculeList->mol_index_from_id(i);
    if(m >= 0) {
      retval = TRUE;
      if(yn)
        moleculeList->show(m);
      else
        moleculeList->hide(m);
    } else
      msgErr<<"Cannot set displayed status of molecule with ID "<<i <<sendmsg;
    return retval;
  }

int CmdMolOn::do_execute(void) {
  int retval;
  int changed = 0;

  if(retval = (moleculeList != NULL)) {
    if(find_mol_list(whichMol)) {
      // working backwards just for the heck of it
      for(int i=(numMol-1); i >= 0; i--) {
	retval = retval &&
			do_action((moleculeList->molecule(idList[i]))->id());
        if(retval)
          changed ++;
      }
      msgInfo << "Changed " << changed << " molecules." << sendmsg;
    } else {
      return FALSE;
    }
  }
  return retval;
}

void CmdMolOn::create_text(void) {
  *cmdText << "mol " << (yn ? "on " : "off ") << whichMol << ends;
}

CmdMolOn::CmdMolOn(char *wm, int newyn, int newUIid)
	: CmdMolecule(wm, Command::MOL_ON, newUIid) {
  yn = newyn;
}


///////////////  make the Nth molecule 'top'.  If more than one molecule
// is specified, make the the first one in the list 'top'.
int CmdMolTop::do_execute(void) {
  int retval;

  if(retval=(moleculeList!=NULL) && find_mol_list(whichMol))
    moleculeList->make_top(idList[0]);
  return retval;
}

void CmdMolTop::create_text(void) {
  *cmdText << "mol top " << whichMol << ends;
}

CmdMolTop::CmdMolTop(char *wm, int newUIid)
	: CmdMolecule(wm, Command::MOL_TOP, newUIid) { }


///////////// set the default atom selection in moleculeList
  int CmdMolSelect::do_execute(void) {
    if (moleculeList == NULL ) {
       return FALSE;
    }
    if (!sel) {
       msgInfo << "Current atom selection:";
       msgInfo << "\n  '" << moleculeList->selection() << "'" << sendmsg;
       return TRUE;
    }
    int retval = moleculeList->set_selection(sel);
    // all this has to do is parse correctly
    if (retval == AtomSel::NO_PARSE) {
       msgErr << "Unable to change current selection." << sendmsg;
       return FALSE;
    }
    return TRUE;
  }

  void CmdMolSelect::create_text(void) {
    *cmdText << "mol selection ";
    if(sel)
      *cmdText << sel;
    *cmdText << ends;
  }

  CmdMolSelect::CmdMolSelect(char *newsel, int newUIid)
    : Command(Command::MOL_SELECT,newUIid) {
    sel = (newsel ? stringdup(newsel) : NULL);
  }
  
  CmdMolSelect::~CmdMolSelect(void) {
    if(sel)  delete [] sel;
  }


///////////// set the default atom representation in moleculeList
 int CmdMolRep::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if(sel) {
        retval = moleculeList->set_representation(sel);
	if(!retval)
	  msgErr << "Unable to change current representation setting."
	         << sendmsg;
      } else {
        msgInfo << "Current atom representation:";
	msgInfo << "\n  '" << moleculeList->representation() << "'" << sendmsg;
      }
    }
    return retval;
  }

  void CmdMolRep::create_text(void) {
    *cmdText << "mol representation ";
    if(sel)
      *cmdText << sel;
    *cmdText << ends;
  }

  CmdMolRep::CmdMolRep(char *newsel, int newUIid)
    : Command(Command::MOL_REP,newUIid) {
    sel = (newsel ? stringdup(newsel) : NULL);
  }
  
  CmdMolRep::~CmdMolRep(void) {
    if(sel)  delete [] sel;
  }


///////////// set the default atom coloring method in moleculeList
 int CmdMolColor::do_execute(void) {
    int retval;
    if(retval = (moleculeList != NULL)) {
      if(sel) {
        retval = moleculeList->set_color(sel);
	if(!retval)
	  msgErr << "Unable to change current color setting." << sendmsg;
      } else {
        msgInfo << "Current atom coloring method:";
	msgInfo << "\n  '" << moleculeList->color() << "'" << sendmsg;
      }
    }
    return retval;
  }

  void CmdMolColor::create_text(void) {
    *cmdText << "mol color ";
    if(sel)
      *cmdText << sel;
    *cmdText << ends;
  }

  CmdMolColor::CmdMolColor(char *newsel, int newUIid)
    : Command(Command::MOL_COLOR,newUIid) {
    sel = (newsel ? stringdup(newsel) : NULL);
  }
  
  CmdMolColor::~CmdMolColor(void) {
    if(sel)  delete [] sel;
  }


///////////// add a new representation to the active molecules
int CmdMolAddRep::do_execute(void) {
  int i, retval;
  int added = 0;

  if(retval=(moleculeList!=NULL) && find_mol_list(whichMol)) {
    for(i=0; i < numMol; i++) {
       int suc = moleculeList->add_rep(idList[i]);
       if(suc < 0) {
	  msgErr << "Unable to add new representation to molecule ";
	  msgErr << (moleculeList->molecule(i))->id() << "." << sendmsg;
       } else {
	  added++;
       }
    }
    msgInfo << "Added current representation to " << added << " molecules.";
    msgInfo << sendmsg;
  }
  return retval;
}

void CmdMolAddRep::create_text(void) {
  *cmdText << "mol addrep " << whichMol << ends;
}

CmdMolAddRep::CmdMolAddRep(char *wm, int newUIid)
	: CmdMolecule(wm, Command::MOL_ADDREP,newUIid) { }


///////////// change a representation for the specified molecule
// NOTE: this can ONLY work on one molecule; if more than one is given, an
// error is printed.
int CmdMolChangeRep::do_execute(void) {
  int retval;

  if(retval=(moleculeList!=NULL) && find_mol_list(whichMol)) {
    if(numMol != 1) {
      msgErr << "Exactly one molecule & representation can be affected by ";
      msgErr << "this command.  Specify an ID or 'top'." << sendmsg;
      return FALSE;
    }
   
    retval = moleculeList->change_rep(repn, idList[0]);

    if(!retval) {
      msgErr << "Illegal mol representation index " << repn;
      msgErr << " for molecule " << (moleculeList->molecule(idList[0]))->id();
      msgErr << sendmsg;
    }
  }
  return retval;
}

void CmdMolChangeRep::create_text(void) {
  *cmdText << "mol modrep " << repn << " " << whichMol << ends;
}

CmdMolChangeRep::CmdMolChangeRep(int rpos, char *wm, int newUIid)
	: CmdMolecule(wm, Command::MOL_MODREP,newUIid) {
  repn = rpos;
}


///////////// change 1 representation characteristic for the specified mol
// NOTE: this can ONLY work on one molecule; if more than one is given, an
// error is printed.
int CmdMolChangeRepItem::do_execute(void) {
  int retval;

  if(retval=(moleculeList!=NULL) && find_mol_list(whichMol)) {
    if(numMol != 1) {
      msgErr << "Exactly one molecule & representation can be affected by ";
      msgErr << "this command.  Specify an ID or 'top'." << sendmsg;
      return FALSE;
    }

    if(repData == COLOR)
      retval = moleculeList->change_repcolor(repn, idList[0], str);
    else if(repData == REP)
      retval = moleculeList->change_repmethod(repn, idList[0], str);
    else if(repData == SEL)
      retval = moleculeList->change_repsel(repn, idList[0], str);

    if(!retval) {
      msgErr << "Cannot change representation";
      msgErr << " for molecule " << (moleculeList->molecule(idList[0]))->id();
      msgErr << sendmsg;
    }
  }
  return retval;
}

void CmdMolChangeRepItem::create_text(void) {
  *cmdText << "mol mod";
  if(repData == COLOR)
    *cmdText << "color ";
  else if(repData == REP)
    *cmdText << "style ";
  else if(repData == SEL)
    *cmdText << "select ";
  *cmdText << repn << " " << whichMol << " " << str << ends;
}

CmdMolChangeRepItem::CmdMolChangeRepItem(int rpos, char *wm,
  		RepData rd, char *s, int newUIid)
	: CmdMolecule(wm, Command::MOL_MODREPITEM, newUIid) {
  repn = rpos;
  repData = rd;
  str = stringdup(s);
}

CmdMolChangeRepItem::~CmdMolChangeRepItem(void) {
  if(str)  delete [] str;
}


///////////// delete a representation for the specified molecule
// NOTE: this can ONLY work on one molecule; if more than one is given, an
// error is printed.
int CmdMolDeleteRep::do_execute(void) {
  int retval;

  if(retval=(moleculeList!=NULL) && find_mol_list(whichMol)) {
    if(numMol != 1) {
      msgErr << "Exactly one molecule & representation can be affected by ";
      msgErr << "this command.  Specify an ID or 'top'." << sendmsg;
      return FALSE;
    }

    retval = moleculeList->del_rep(repn, idList[0]);

    if(!retval) {
      msgErr << "Illegal mol representation index " << repn;
      msgErr << " for molecule " << (moleculeList->molecule(idList[0]))->id();
      msgErr << sendmsg;
    }
  }
  return retval;
}

void CmdMolDeleteRep::create_text(void) {
  *cmdText << "mol delrep " << repn << " " << whichMol << ends;
}

CmdMolDeleteRep::CmdMolDeleteRep(int rpos, char *wm, int newUIid)
	: CmdMolecule(wm, Command::MOL_DELREP, newUIid) {
  repn = rpos;
}


