/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: CmdSigma.C,v $
 *	$Author: leech $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 1995/05/16 21:08:35 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Command objects for affecting Sigma interface
 *
 ***************************************************************************/

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "CmdSigma.h"
#include "CommandQueue.h"
#include "Global.h"
#include "utilities.h"

// Sigma library include files
#include <sigma/mdsocket.h>
#include <sigma/mdbuffer.h>
#include <sigma/Communication.h>
#include <sigma/Graphics.h>
#include <sigma/Id.h>


// the following defines commands for the Cmdtypes:
// SIGMA_EVENT

////////////////////////////////////////////////////////////////////
///////////////////////  text processors
////////////////////////////////////////////////////////////////////

// text callback routine for 'sigma'; return TRUE if an error occurs.
// sigma commands are:
//      sigma ev         - send event 'ev'
//      sigma ev param   - send event 'ev' with numeric argument 'param'
//      sigma ev 'id' object elem - send event 'ev' with Id arg (object,elem)
//      sigma ev 'pt' x y z - send event 'ev' with Point3 argument (x,y,z)
// returns the value for texterr -- FALSE means all okay
int text_cmd_sigma(int argc, char **argv, CommandQueue *cmdQueue, int id) {

  int event;
  if (argc < 2 || argc > 6 || sscanf(argv[1], "%d", &event) != 1) {
    msgErr << "Expecting 'sigma event-num [param-num|id # #|pt x y z]'";
    msgErr << sendmsg;
    return TRUE;
  }

  if (argc == 2) {
    cmdQueue->append(new CmdSigmaEvent(event, id));
    return FALSE;
  }

  if (argc == 3) {
    int param = atoi(argv[2]);
    cmdQueue->append(new CmdSigmaEvent(event, param, id));
    return FALSE;
  }

  if (argc == 5 && !strupncmp(argv[2], "id", CMDLEN)) {
    int object, elem;

    object = atoi(argv[3]);
    elem = atoi(argv[4]);
    cmdQueue->append(new CmdSigmaEvent(event, object, elem, id));
    return FALSE;
  }

  if (argc == 6 && !strupncmp(argv[2], "pt", CMDLEN)) {
    float x, y, z;

    x = atof(argv[3]);
    y = atof(argv[4]);
    z = atof(argv[5]);
    cmdQueue->append(new CmdSigmaEvent(event, x, y, z, id));
    return FALSE;
  }

  msgErr << "Unrecognized arguments to 'sigma' command" << sendmsg;
  msgErr << "Expecting args: event-num [param-num|id # #|pt x y z]";
  msgErr << sendmsg;
  return TRUE;
}


///////////////////// Send an event to Sigma
int CmdSigmaEvent::do_execute(void) {
  int retval = 1; // success

  msgInfo << "CmdSigma::do_event ";

  switch (paramType) {
    case EV_NO_PARAM:
      msgInfo << "event = " << event << sendmsg;
      buffer_send_event(event);
      break;
    case EV_INT_PARAM:
      msgInfo << "event = " << event << " param = " << param << sendmsg;
      buffer_send_event(event, param);
      break;
    case EV_ID_PARAM: {
	Id id(object, element);
	msgInfo << "event = " << event << " Id object = " << object
		<< " element = " << element << sendmsg;
	buffer_send_event(event, &id);
      }
      break;
    case EV_POINT3_PARAM: {
	Point3 pt;
	pt.x = x;
	pt.y = y;
	pt.z = z;
	msgInfo << "event = " << event << " point = " << x << " "
		<< y << " " << z << sendmsg;
	buffer_send_event(event, &pt);
      }
      break;
    default:
      msgErr << "Unknown paramType " << paramType << ". No message sent!"
	     << sendmsg;
      retval = 0;
      break;
  }
  return retval;
}


void CmdSigmaEvent::create_text(void) {
  *cmdText << "sigma " << event;
  switch (paramType) {
    case EV_INT_PARAM: {
	*cmdText << " " << param;
      }
      break;
    case EV_ID_PARAM: {
	*cmdText << " id " << object << " " << element;
      }
      break;
    case EV_POINT3_PARAM: {
	*cmdText << " pt " << x << " " << y << " " << z;
      }
      break;
  }
  *cmdText << ends;
}


// constructor: event code
CmdSigmaEvent::CmdSigmaEvent(int ev, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
  paramType = EV_NO_PARAM;
  event = ev;
}

// constructor: event code, numeric parameter
CmdSigmaEvent::CmdSigmaEvent(int ev, int p, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
  paramType = EV_INT_PARAM;
  event = ev;
  param = p;
}

// constructor: event code, Id object, Id element
CmdSigmaEvent::CmdSigmaEvent(int ev, int obj, int elem, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
  paramType = EV_ID_PARAM;
  event = ev;
  object = obj;
  element = elem;
}

// constructor: event code, coordinate x,y,z
CmdSigmaEvent::CmdSigmaEvent(int ev, float xc, float yc, float zc, int newUIid)
	: Command(Command::SIGMA_EVENT, newUIid) {
  paramType = EV_POINT3_PARAM;
  event = ev;
  x = xc;
  y = yc;
  z = zc;
}


