/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DispCmds.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.9 $	$Date: 1995/05/11 22:37:55 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * DispCmds - different display commands which take data and put it in
 *	a storage space provided by a given Displayable object.
 *
 * Notes:
 *	1. All coordinates are stored as 3 points (x,y,z), even if meant
 * for a 2D object.  The 3rd coord for 2D objects will be ignored.
 ***************************************************************************/

#include <string.h>
#include <math.h>
#include "DispCmds.h"
#include "utilities.h"

//////////////////////  base class constructor
DispCmd::DispCmd(int code, int size) { 
  cmdCode = code;
  cmdSize = size;
  Data = dataLoc = NULL; 
}

//////////////////////  base class destructor
DispCmd::~DispCmd(void) {
  // does nothing
}


// do the action
void DispCmd::put(Displayable *dobj) {
   if (!dobj->start_cmd(cmdCode, cmdSize)) {
      return;          // abort if I cannot add this command
   }
   dataLoc = dobj->cmdListPos;          // get the new data...
   if(cmdSize > 0 && Data && dataLoc) {
      memcpy(dataLoc, Data, cmdSize);   // .. and copy it to the disp list
   }
   dobj->end_cmd();                     // fix up the pointers
}

// do the action AGAIN, if we've done it before; if not, same as put
void DispCmd::reput(Displayable *dobj) {
  if(!dataLoc)
    DispCmd::put(dobj);
  else if(cmdSize > 0 && Data)
    memcpy(dataLoc, Data, cmdSize);
}


///////////////////////  DispCmd derived classes  

//*************************************************************

// create a block of floating-point data values to be used in drawing
// This version adds a new block of data to the current data blocks
  // DOES NOT HAVE AN EMPTY CONSTRUCTOR
#ifdef VMDCAVE
DispCmdData::DispCmdData(int n, float *d) 
  : DispCmd(DMOREDATA, n * sizeof(float)) {
    Data = (void *)d;
  }
#else
DispCmdData::DispCmdData(int n, float *d) 
 : DispCmd(DMOREDATA, sizeof(float *)) {
    Data = (void *)&dptr;
    dptr = d;
    if(n);
  }
#endif

  // give new data, and just 'put' the command in again
void DispCmdData::putdata(float *d, Displayable *dobj) {
#ifdef VMDCAVE
    Data = (void *)d;
#else
    dptr = d;
#endif
    DispCmd::put(dobj);
  }

  // put in new data, and reput the command
void DispCmdData::reputdata(float *d, Displayable *dobj) {
#ifdef VMDCAVE
    Data = (void *)d;
#else
    dptr = d;
#endif
    DispCmd::reput(dobj);
}


//*************************************************************

// create a block of floating-point data values to be used in drawing
// This version make the given data block cancel out the usage of any
// previously current data blocks

  // DOES NOT HAVE AN EMPTY CONSTRUCTOR
#ifdef VMDCAVE
  DispCmdNewData::DispCmdNewData(int n, float *d) 
  : DispCmd(DNEWDATA, n * sizeof(float)) {
    Data = (void *)d;
  }
#else
  DispCmdNewData::DispCmdNewData(int n, float *d) 
  : DispCmd(DNEWDATA, sizeof(float *)) {
    Data = (void *)&dptr;
    dptr = d;
    if(n);
  }
#endif
  
  // give new data, and just 'put' the command in again
void DispCmdNewData::putdata(float *d, Displayable *dobj) {
#ifdef VMDCAVE
    Data = (void *)d;
#else
    dptr = d;
#endif
    DispCmd::put(dobj);
}

  // put in new data, and reput the command
void DispCmdNewData::reputdata(float *d, Displayable *dobj) {
#ifdef VMDCAVE
    Data = (void *)d;
#else
    dptr = d;
#endif
    DispCmd::reput(dobj);
}


//*************************************************************

// command to clear the display
  DispCmdClear::DispCmdClear(void) : DispCmd(DCLEAR, 0) { }


//*************************************************************

// command to push the display stacks
DispCmdPush::DispCmdPush(void) : DispCmd(DPUSH, 0) { }


//*************************************************************

// command to pop the display stacks
DispCmdPop::DispCmdPop(void) : DispCmd(DPOP, 0) { }

//*************************************************************

// command to replace the transformation stack with the given matrix
  // empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdLoad::DispCmdLoad(void) : DispCmd(DLOAD, 4*4*sizeof(float)) { }

DispCmdLoad::DispCmdLoad(Matrix4 *newmat) : DispCmd(DLOAD, 4*4*sizeof(float)) {
  Data = (void *)(newmat->mat);
}

// put in new data, and put the command
void DispCmdLoad::putdata(Matrix4 *newmat, Displayable *dobj) {
  Data = (void *)(newmat->mat);
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdLoad::reputdata(Matrix4 *newmat, Displayable *dobj) {
  Data = (void *)(newmat->mat);
  DispCmd::reput(dobj);
}


//*************************************************************

// command to multiply the transformation stack with the given matrix
  // empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdMult::DispCmdMult(void) : DispCmd(DMULT, 4*4*sizeof(float)) { }

DispCmdMult::DispCmdMult(Matrix4 *newmat) : DispCmd(DMULT, 4*4*sizeof(float)) {
  Data = (void *)(newmat->mat);
}

// put in new data, and put the command
void DispCmdMult::putdata(Matrix4 *newmat, Displayable *dobj) {
  Data = (void *)(newmat->mat);
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdMult::reputdata(Matrix4 *newmat, Displayable *dobj) {
  Data = (void *)(newmat->mat);
  DispCmd::reput(dobj);
}

//*************************************************************

// command to do a translation
  // empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdTrans::DispCmdTrans(void) : DispCmd(DTRANS, 3*sizeof(float)) { }

DispCmdTrans::DispCmdTrans(float x, float y, float z) 
      : DispCmd(DTRANS, 3*sizeof(float)) {
  Data = (void *)tr;
  tr[0] = x;  tr[1] = y;  tr[2] = z;
}

// put in new data, and put the command
void DispCmdTrans::putdata(float x, float y, float z, Displayable *dobj) {
  tr[0] = x;  tr[1] = y;  tr[2] = z;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdTrans::reputdata(float x, float y, float z, Displayable *dobj) {
  tr[0] = x;  tr[1] = y;  tr[2] = z;
  DispCmd::reput(dobj);
}


//*************************************************************

// command to do a scaling
  // empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdScale::DispCmdScale(void) : DispCmd(DSCALE, 3*sizeof(float)) {
  Data = (void *)tr;
}

DispCmdScale::DispCmdScale(float x, float y, float z) 
      : DispCmd(DSCALE, 3*sizeof(float)) {
  Data = (void *)tr;
  tr[0] = x;  tr[1] = y;  tr[2] = z;
}
DispCmdScale::DispCmdScale(float s) : DispCmd(DSCALE, 3*sizeof(float)) {
  Data = (void *)tr;
  tr[0] = tr[1] = tr[2] = s;
}

// put in new data, and put the command
void DispCmdScale::putdata(float x, float y, float z, Displayable *dobj) {
  tr[0] = x;  tr[1] = y;  tr[2] = z;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdScale::reputdata(float x, float y, float z, Displayable *dobj) {
  tr[0] = x;  tr[1] = y;  tr[2] = z;
  DispCmd::reput(dobj);
}

//*************************************************************

// command to do a rotation about a specific axis
  // empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdRot::DispCmdRot(void) : DispCmd(DROT, 2*sizeof(float)) {
  Data = (void *)tr;
}

DispCmdRot::DispCmdRot(float t, char a) : DispCmd(DROT, 2*sizeof(float)) {
  Data = (void *)tr;
  tr[0] = t;
  tr[1] = 0.0 + (float)(a - 'x');
}

// put in new data, and put the command
void DispCmdRot::putdata(float t, char a, Displayable *dobj) {
  tr[0] = t;
  tr[1] = 0.0 + (float)(a - 'x');
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdRot::reputdata(float t, char a, Displayable *dobj) {
  tr[0] = t;
  tr[1] = 0.0 + (float)(a - 'x');
  DispCmd::reput(dobj);
}

//*************************************************************

// plot a point at the given position
  // empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdPoint::DispCmdPoint(void) : DispCmd(DPOINT, 3*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdPoint::DispCmdPoint(float *newpos) : DispCmd(DPOINT, 3*sizeof(float)) {
  Data = (void *)pos;
  memcpy(Data, (void *)newpos, cmdSize);
}

// put in new data, and put the command
void DispCmdPoint::putdata(float *newpos, Displayable *dobj) {
  memcpy(Data, (void *)newpos, cmdSize);
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdPoint::reputdata(float *newpos, Displayable *dobj) {
  memcpy(Data, (void *)newpos, cmdSize);
  DispCmd::reput(dobj);
}



// plot a point at the given position, using indices
  // empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdPointIndex::DispCmdPointIndex(void) : DispCmd(DPOINT_I, sizeof(int)) {
  Data = (void *)&pos;
}

DispCmdPointIndex::DispCmdPointIndex(int newpos) 
 : DispCmd(DPOINT_I, sizeof(int)) {
  Data = (void *)&pos;
  pos = newpos;
}

// put in new data, and put the command
void DispCmdPointIndex::putdata(int newpos, Displayable *dobj) {
  pos = newpos;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdPointIndex::reputdata(int newpos, Displayable *dobj) {
  pos = newpos;
  DispCmd::reput(dobj);
}


//*************************************************************

// plot a sphere of specified radius at the given position
  // empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdSphere::DispCmdSphere(void) : DispCmd(DSPHERE, 4*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdSphere::DispCmdSphere(float *newpos, float rad) 
: DispCmd(DSPHERE, 4*sizeof(float)) {
  Data = (void *)pos;
  memcpy(Data, (void *)newpos, 3*sizeof(float));
  pos[3] = rad;
}

// put in new data, and put the command
void DispCmdSphere::putdata(float *newpos, float rad, Displayable *dobj) {
  memcpy(Data, (void *)newpos, 3*sizeof(float));
  pos[3] = rad;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdSphere::reputdata(float *newpos, float rad, Displayable *dobj) {
  memcpy(Data, (void *)newpos, 3*sizeof(float));
  pos[3] = rad;
  DispCmd::reput(dobj);
}


// plot a sphere of specified radius at the given position, using indices
DispCmdSphereIndex::DispCmdSphereIndex(void) 
: DispCmd(DSPHERE_I, 2*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdSphereIndex::DispCmdSphereIndex(int newpos, float rad)
  : DispCmd(DSPHERE_I, 2*sizeof(float)) {
  Data = (void *)pos;
  pos[0] = (float)newpos;
  pos[1] = rad;
}

// put in new data, and put the command
void DispCmdSphereIndex::putdata(int newpos, float rad, Displayable *dobj) {
  pos[0] = (float)newpos;
  pos[1] = rad;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdSphereIndex::reputdata(int newpos, float rad, Displayable *dobj) {
  pos[0] = (float)newpos;
  pos[1] = rad;
  DispCmd::reput(dobj);
}

//*************************************************************

// plot a line at the given position
DispCmdLine::DispCmdLine(void) : DispCmd(DLINE, 6*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdLine::DispCmdLine(float *pos1, float *pos2) 
: DispCmd(DLINE, 6*sizeof(float)) {
  Data = (void *)pos;
  memcpy((void *)(pos), (void *)pos1, 3*sizeof(float));
  memcpy((void *)(pos + 3), (void *)pos2, 3*sizeof(float));
}

// put in new data, and put the command
void DispCmdLine::putdata(float *pos1, float *pos2, Displayable *dobj) {
   memcpy((void *)(pos), (void *)pos1, 3*sizeof(float));
   memcpy((void *)(pos + 3), (void *)pos2, 3*sizeof(float));
   DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdLine::reputdata(float *pos1, float *pos2, Displayable *dobj) {
  memcpy((void *)(pos), (void *)pos1, 3*sizeof(float));
  memcpy((void *)(pos + 3), (void *)pos2, 3*sizeof(float));
  DispCmd::reput(dobj);
}


// plot a line at the given position, using indices
DispCmdLineIndex::DispCmdLineIndex(void) : DispCmd(DLINE_I, 2*sizeof(int)) {
  Data = (void *)pos;
}

DispCmdLineIndex::DispCmdLineIndex(int pos1, int pos2) 
: DispCmd(DLINE_I, 2*sizeof(int)) {
  Data = (void *)pos;
  pos[0] = pos1;
  pos[1] = pos2;
}

// put in new data, and put the command
void DispCmdLineIndex::putdata(int pos1, int pos2, Displayable *dobj) {
  pos[0] = pos1;
  pos[1] = pos2;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdLineIndex::reputdata(int pos1, int pos2, Displayable *dobj) {
  pos[0] = pos1;
  pos[1] = pos2;
  DispCmd::reput(dobj);
}

//*************************************************************
// draw a triangle

// given the vectors, compute the normal, then set the pos array
void DispCmdTriangle::do_calc(float *pos1, float *pos2, float *pos3) {
  int i;
  float tmp1[3], tmp2[3], tmp3[3];  // precompute the normal for
  for (i=0; i<3; i++) {             //   faster drawings later
     tmp1[i] = pos2[i] - pos1[i];
     tmp2[i] = pos3[i] - pos2[i];
  }
  cross_prod( tmp3, tmp1, tmp2);  
  normalize(tmp3);
  set_array(pos1, pos2, pos3, tmp3, tmp3, tmp3);
}

// set up the data for the DTRIANGLE drawing command
void DispCmdTriangle::set_array(float *pos1, float *pos2, float *pos3,
		  float *norm1, float *norm2, float *norm3)
{
  // coordinates first
  memcpy( (void *)(pos + 0), (void *) pos1, 3*sizeof(float));
  memcpy( (void *)(pos + 3), (void *) pos2, 3*sizeof(float));
  memcpy( (void *)(pos + 6), (void *) pos3, 3*sizeof(float));
  // then the normals (all are the same)
  memcpy( (void *)(pos + 9), (void *) norm1, 3*sizeof(float));
  memcpy( (void *)(pos +12), (void *) norm2, 3*sizeof(float));
  memcpy( (void *)(pos +15), (void *) norm3, 3*sizeof(float));
}

DispCmdTriangle::DispCmdTriangle(void) : DispCmd(DTRIANGLE, 18*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdTriangle::DispCmdTriangle(float *pos1, float *pos2, float *pos3)
      : DispCmd(DTRIANGLE, 18*sizeof(float)) {
  Data = (void *) pos;
  do_calc(pos1, pos2, pos3);
}
DispCmdTriangle::DispCmdTriangle(float *pos1, float *pos2, float *pos3,
				 float *norm1, float *norm2, float *norm3)
      : DispCmd(DTRIANGLE, 18*sizeof(float)) {
  Data = (void *) pos;
  float temp[3][3];
  copy(temp[0], norm1);
  normalize(temp[0]);
  copy(temp[1], norm2);
  normalize(temp[1]);
  copy(temp[2], norm3);
  normalize(temp[2]);
  set_array(pos1, pos2, pos3, temp[0], temp[1], temp[2]);
}

// put in new data, and put the command
void DispCmdTriangle::putdata(float *pos1, float *pos2, 
                              float *pos3, Displayable *dobj) {
  do_calc(pos1, pos2, pos3);
  DispCmd::put(dobj);
}
void DispCmdTriangle::putdata(float *pos1, float *pos2, float *pos3,
			      float *norm1, float *norm2, float *norm3,
                              Displayable *dobj) {
  float temp[3][3];
  copy(temp[0], norm1);
  normalize(temp[0]);
  copy(temp[1], norm2);
  normalize(temp[1]);
  copy(temp[2], norm3);
  normalize(temp[2]);
  set_array(pos1, pos2, pos3, temp[0], temp[1], temp[2]);
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdTriangle::reputdata(float *pos1, float *pos2, 
                                float *pos3, Displayable *dobj) {
  do_calc(pos1, pos2, pos3);
  DispCmd::reput(dobj);
}
void DispCmdTriangle::reputdata(float *pos1, float *pos2, float *pos3,
			      float *norm1, float *norm2, float *norm3,
                              Displayable *dobj) {
  float temp[3][3];
  copy(temp[0], norm1);
  normalize(temp[0]);
  copy(temp[1], norm2);
  normalize(temp[1]);
  copy(temp[2], norm3);
  normalize(temp[2]);
  set_array(pos1, pos2, pos3, temp[0], temp[1], temp[2]);
   DispCmd::reput(dobj);
}


// draw a triangle, given the three points and normal, using indices
DispCmdTriangleIndex::DispCmdTriangleIndex(void) 
: DispCmd(DTRIANGLE_I, 4*sizeof(int)) {
  Data = (void *)pos;
}

DispCmdTriangleIndex::DispCmdTriangleIndex(int norml, int pos1, int pos2, 
                                            int pos3)
      : DispCmd(DTRIANGLE_I, 4*sizeof(int)) {
  Data = (void *)pos;
  pos[0] = norml;  pos[1] = pos1;  pos[2] = pos2;  pos[3] = pos3;
}

// put in new data, and put the command
void DispCmdTriangleIndex::putdata(int norml, int pos1, int pos2, int pos3, 
                                    Displayable *dobj) {
  pos[0] = norml;  pos[1] = pos1;  pos[2] = pos2;  pos[3] = pos3;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdTriangleIndex::reputdata(int norml, int pos1, int pos2, int pos3, 
                                      Displayable *dobj) {
  pos[0] = norml;  pos[1] = pos1;  pos[2] = pos2;  pos[3] = pos3;
  DispCmd::reput(dobj);
}


//*************************************************************

// draw a square, given 3 of four points
void DispCmdSquare::do_calc(float *pos1, float *pos2, float *pos3) {
  int i;
  float tmp1[3], tmp2[3], tmp3[3];  // precompute the normal for
  for (i=0; i<3; i++) {             //   faster drawings later
     tmp1[i] = pos2[i] - pos1[i];
     tmp2[i] = pos3[i] - pos2[i];
  }
  cross_prod( tmp3, tmp1, tmp2);  
  normalize(tmp3);

  memcpy( (void *) pos     , (void *) tmp3, 3*sizeof(float)); // the normal
  memcpy( (void *)(pos + 3), (void *) pos1, 3*sizeof(float));
  memcpy( (void *)(pos + 6), (void *) pos2, 3*sizeof(float));
  memcpy( (void *)(pos + 9), (void *) pos3, 3*sizeof(float));
  for (i=0; i<3; i++)
    tmp3[i] = pos1[i] + tmp2[i];  // compute the fourth point
  memcpy( (void *)(pos + 12), (void *) tmp3, 3*sizeof(float));
}

// empty constructor ... must be used with 'putdata' or 'reputdata'
DispCmdSquare::DispCmdSquare(void) : DispCmd(DSQUARE, 15*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdSquare::DispCmdSquare(float *pos1, float *pos2, float *pos3)
      : DispCmd(DSQUARE, 15*sizeof(float)) {
  Data = (void *) pos;     // set up the data for the DSQUARE command
  do_calc(pos1, pos2, pos3);
}

// put in new data, and put the command
void DispCmdSquare::putdata(float *pos1, float *pos2, float *pos3, 
                             Displayable *dobj) {
  do_calc(pos1, pos2, pos3);
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdSquare::reputdata(float *pos1, float *pos2, float *pos3, 
                               Displayable *dobj) {
  do_calc(pos1, pos2, pos3);
  DispCmd::reput(dobj);
}



// draw a triangle, given four points and normal, using indices
DispCmdSquareIndex::DispCmdSquareIndex(void) 
: DispCmd(DSQUARE_I, 4*sizeof(int)) {
  Data = (void *)pos;
}

DispCmdSquareIndex::DispCmdSquareIndex(int norml, int pos1, int pos2, 
                                        int pos3, int pos4)
      : DispCmd(DSQUARE_I, 4*sizeof(int)) {
  Data = (void *)pos;
  pos[0] = norml;
  pos[1] = pos1;  pos[2] = pos2;  pos[3] = pos3;  pos[4] = pos4;
}

// put in new data, and put the command
void DispCmdSquareIndex::putdata(int norml, int pos1, int pos2, int pos3, 
                                  int pos4, Displayable *dobj){
  pos[0] = norml;
  pos[1] = pos1;  pos[2] = pos2;  pos[3] = pos3;  pos[4] = pos4;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdSquareIndex::reputdata(int norml, int pos1, int pos2, int pos3, 
                                     int pos4, Displayable *dobj){
  pos[0] = norml;
  pos[1] = pos1;  pos[2] = pos2;  pos[3] = pos3;  pos[4] = pos4;
  DispCmd::reput(dobj);
}

//*************************************************************

// plot a cylinder at the given position
DispCmdCylinder::DispCmdCylinder(void) : DispCmd(DCYLINDER, 8*sizeof(float)) {
#ifndef USE_SLOW_CYLINDERS
  pos = NULL;
  lastres = 0;
#endif
  Data = (void *)pos;
}

// this is used to precalculate the cylinder data for speedup
// in renderers without hardware cylinders.  For example, the GL
// library.  There are res number of edges (with a norm, and two points)
void DispCmdCylinder::make_cylinder_info(float *pos1, float *pos2,
					 float rad, int res)
{
   float lenaxis[3];
   subtract(lenaxis, pos1, pos2);  // check that it's valid
   if (dot_prod(lenaxis,lenaxis) == 0.0 || res <= 0) {
      return;
   }
#ifndef USE_SLOW_CYLINDERS
   if (lastres < res || pos == NULL) {   // check that I have enough space
      if (pos) delete [] pos;
      pos = new float[8 + res*3*3];  // 8 for the input data, 3*(x,y,z)
      lastres = res;
      cmdSize = (8 + res*3*3 )* sizeof(float);
   }
#endif
   Data = (void *)pos;  // easy enough to do this here every time for every
                        // call instead of putting this everywhere

   // put in the simple stuff
   memcpy((void *)pos, (void *)pos1, 3*sizeof(float));
   memcpy((void *)(pos + 3), (void *)pos2, 3*sizeof(float));
   pos[6] = rad;
   pos[7] = (float)res;

#ifndef USE_SLOW_CYLINDERS
   // and the precompute info
   float axis[3];
   subtract(axis, pos1, pos2);
   normalize(axis);
   int i;  // find an axis not aligned with the cylinder
   if (fabs(axis[0]) < fabs(axis[1]) &&
       fabs(axis[0]) < fabs(axis[2])) {
      i = 0;
   } else if (fabs(axis[1]) < fabs(axis[2])) {
      i = 1;
   } else {
      i = 2;
   }
   float perp[3];
   perp[i] = 0;                    // this is not aligned with the cylinder
   perp[(i+1)%3] = axis[(i+2)%3];
   perp[(i+2)%3] = -axis[(i+1)%3];
   normalize(perp);
   float perp2[3];
   cross_prod(perp2, axis, perp); // find a normal to the cylinder

   float hresinc = TWOPI / ((float) res);
   float theta = 0.0;
   float *posptr = pos+8;
   float m, n;
   for (int h=0; h<res; h++) {
      m = cosf(theta);
      n = sinf(theta);
      posptr[0] = m*perp[0] + n*perp2[0]; // add the normal
      posptr[1] = m*perp[1] + n*perp2[1];
      posptr[2] = m*perp[2] + n*perp2[2];

      posptr[3] = pos2[0] + rad * posptr[0]; // start
      posptr[4] = pos2[1] + rad * posptr[1];
      posptr[5] = pos2[2] + rad * posptr[2];
      posptr[6] = posptr[3] + lenaxis[0];  // and end of the edge
      posptr[7] = posptr[4] + lenaxis[1];
      posptr[8] = posptr[5] + lenaxis[2];
      posptr += 9;
      theta += hresinc;
   }
#endif
}
      

DispCmdCylinder::DispCmdCylinder(float *pos1, float *pos2, float rad, int res) 
      : DispCmd(DCYLINDER, 8*sizeof(float))
{
#ifndef USE_SLOW_CYLINDERS
   pos = NULL;
   lastres = 0;
#endif
   make_cylinder_info(pos1, pos2, rad, res);
}

// put in new data, and put the command
void DispCmdCylinder::putdata(float *pos1,float *pos2,float rad,int res,
                               Displayable *dobj)
{
   make_cylinder_info(pos1, pos2, rad, res);
   DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdCylinder::reputdata(float *pos1,float *pos2,float rad,int res,
                                 Displayable *dobj) {
   make_cylinder_info(pos1, pos2, rad, res);
   DispCmd::reput(dobj);
}


// plot a cylinder at the given position, using indices
DispCmdCylinderIndex::DispCmdCylinderIndex(void) 
: DispCmd(DCYLINDER_I, 4*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdCylinderIndex::DispCmdCylinderIndex(int pos1, int pos2, float rad, 
                                           int res) 
      : DispCmd(DCYLINDER_I, 4*sizeof(float)) {
  Data = (void *)pos;
  pos[0] = (float)pos1;	pos[1] = (float)pos2;
  pos[2] = rad;		pos[3] = (float)res;
}

// put in new data, and put the command
void DispCmdCylinderIndex::putdata(int pos1, int pos2, float rad, int res, 
Displayable *dobj) {
  pos[0] = (float)pos1;	pos[1] = (float)pos2;
  pos[2] = rad;		pos[3] = (float)res;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdCylinderIndex::reputdata(int pos1, int pos2, float rad, int res, 
Displayable *dobj) {
  pos[0] = (float)pos1;	pos[1] = (float)pos2;
  pos[2] = rad;		pos[3] = (float)res;
  DispCmd::reput(dobj);
}

//*************************************************************
// plot a cone at the given position
DispCmdCone::DispCmdCone(void) : DispCmd(DCONE, 8*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdCone::DispCmdCone(float *pos1, float *pos2, float rad, int res) 
      : DispCmd(DCONE, 8*sizeof(float)) {
  Data = (void *)pos;
  memcpy((void *)pos, (void *)pos1, 3*sizeof(float));
  memcpy((void *)(pos + 3), (void *)pos2, 3*sizeof(float));
  pos[6] = rad;
  pos[7] = (float)res;
}

// put in new data, and put the command
void DispCmdCone::putdata(float *pos1,float *pos2,float rad,int res,
Displayable *dobj) {
  memcpy((void *)pos, (void *)pos1, 3*sizeof(float));
  memcpy((void *)(pos + 3), (void *)pos2, 3*sizeof(float));
  pos[6] = rad;
  pos[7] = (float)res;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdCone::reputdata(float *pos1,float *pos2,float rad,int res,
Displayable *dobj) {
  memcpy((void *)pos, (void *)pos1, 3*sizeof(float));
  memcpy((void *)(pos + 3), (void *)pos2, 3*sizeof(float));
  pos[6] = rad;
  pos[7] = (float)res;
  DispCmd::reput(dobj);
}

// plot a cone at the given position, using indices
DispCmdConeIndex::DispCmdConeIndex(void) : DispCmd(DCONE_I, 4*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdConeIndex::DispCmdConeIndex(int pos1, int pos2, float rad, int res) 
      : DispCmd(DCONE_I, 4*sizeof(float)) {
  Data = (void *)pos;
  pos[0] = (float)pos1;	pos[1] = (float)pos2;
  pos[2] = rad;		pos[3] = (float)res;
}

// put in new data, and put the command
void DispCmdConeIndex::putdata(int pos1, int pos2, float rad, int res, 
Displayable *dobj) {
  pos[0] = (float)pos1;	pos[1] = (float)pos2;
  pos[2] = rad;		pos[3] = (float)res;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdConeIndex::reputdata(int pos1, int pos2, float rad, int res, 
Displayable *dobj) {
  pos[0] = (float)pos1;	pos[1] = (float)pos2;
  pos[2] = rad;		pos[3] = (float)res;
  DispCmd::reput(dobj);
}

//*************************************************************

// set the current color to the given one in a list; this specifies the index
DispCmdColorIndex::DispCmdColorIndex(void) 
: DispCmd(DCOLORINDEX, sizeof(int)) {
  Data = (void *)(&color);
}

DispCmdColorIndex::DispCmdColorIndex(int newcol) 
: DispCmd(DCOLORINDEX, sizeof(int)) {
  Data = (void *)(&color);
  color = newcol;
}

// put in new data, and put the command
void DispCmdColorIndex::putdata(int newcol, Displayable *dobj) {
  color = newcol;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdColorIndex::reputdata(int newcol, Displayable *dobj) {
  color = newcol;
  DispCmd::reput(dobj);
}

//*************************************************************
// set the current color to the given rgb value
DispCmdColorRGB::DispCmdColorRGB(void) : DispCmd(DCOLORRGB, 3*sizeof(float)) {
  Data = (void *)color;
}

DispCmdColorRGB::DispCmdColorRGB(float *col) 
: DispCmd(DCOLORRGB, 3*sizeof(float)) {
  Data = (void *)color;
  memcpy(Data, (void *)col, cmdSize);
}

// put in new data, and put the command
void DispCmdColorRGB::putdata(float *col, Displayable *dobj) {
  memcpy(Data, (void *)col, cmdSize);
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdColorRGB::reputdata(float *col, Displayable *dobj) {
  memcpy(Data, (void *)col, cmdSize);
  DispCmd::reput(dobj);
}

//*************************************************************

// define values for the Nth color
 DispCmdColorDef::DispCmdColorDef(void) : DispCmd(DCOLORDEF,(1+COLOR_ITEMS) * sizeof(float)) {
  Data = (void *)data;
}

 DispCmdColorDef::DispCmdColorDef(int n, float *cdat)
  : DispCmd(DCOLORDEF, (1 + COLOR_ITEMS) * sizeof(float)) {
  Data = (void *)data;
  data[0] = (float)n;
  memcpy((void *)(data + 1), (void *)cdat, COLOR_ITEMS * sizeof(float));
}

// put in new data, and put the command
void  DispCmdColorDef::putdata(int n, float *cdat, Displayable *dobj) {
  data[0] = (float)n;
  memcpy((void *)(data + 1), (void *)cdat, COLOR_ITEMS * sizeof(float));
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void  DispCmdColorDef::reputdata(int n, float *cdat, Displayable *dobj) {
  data[0] = (float)n;
  memcpy((void *)(data + 1), (void *)cdat, COLOR_ITEMS * sizeof(float));
  DispCmd::reput(dobj);
}

//*************************************************************

// command to turn on or off material characteristics display
DispCmdMaterials::DispCmdMaterials(void) : DispCmd(DMATERIALS, sizeof(int)) {
  Data = (void *)(&onoff);
}

DispCmdMaterials::DispCmdMaterials(int on) : DispCmd(DMATERIALS, sizeof(int)) {
  Data = (void *)(&onoff);
  onoff = on;
}

// put in new data, and put the command
void DispCmdMaterials::putdata(int on, Displayable *dobj) {
  onoff = on;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdMaterials::reputdata(int on, Displayable *dobj) {
  onoff = on;
  DispCmd::reput(dobj);
}

//*************************************************************

// display text at the current text coordinates
// Note: This does NOT make a copy of the string, so make sure the text stays
// around until the 'put' routine is called!!!
DispCmdText::DispCmdText(char *newtxt) : DispCmd(DTEXT,
   strlen(newtxt) + 1 + sizeof(int) - ((strlen(newtxt) + 1) % sizeof(int))) {
  Data = (void *)newtxt;
}

// put in new data, and put the command
// be careful, the text must be the same length as before!
void DispCmdText::putdata(char *newtxt, Displayable *dobj) {
  Data = (void *)newtxt;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
// be careful, the text must be the same length as before!
void DispCmdText::reputdata(char *newtxt, Displayable *dobj) {
  Data = (void *)newtxt;
  DispCmd::reput(dobj);
}

//*************************************************************

// move the current text position to the given position
DispCmdTextPos::DispCmdTextPos(void) : DispCmd(DTEXTPOS, 3*sizeof(float)) {
  Data = (void *)pos;
}

DispCmdTextPos::DispCmdTextPos(float *pos1) 
: DispCmd(DTEXTPOS, 3*sizeof(float)) {
  Data = (void *)pos;
  memcpy((void *)pos, (void *)pos1, cmdSize);
}

// put in new data, and put the command
void DispCmdTextPos::putdata(float *pos1, Displayable *dobj) {
  memcpy((void *)pos, (void *)pos1, cmdSize);
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdTextPos::reputdata(float *pos1, Displayable *dobj) {
  memcpy((void *)pos, (void *)pos1, cmdSize);
  DispCmd::reput(dobj);
}


// move the current text position to the given position, using indices
DispCmdTextPosIndex::DispCmdTextPosIndex(void) 
: DispCmd(DTEXTPOS_I, sizeof(int)) {
  Data = (void *)(&pos);
}

DispCmdTextPosIndex::DispCmdTextPosIndex(int pos1) 
: DispCmd(DTEXTPOS_I, sizeof(int)) {
  Data = (void *)&pos;
  pos = pos1;
}

// put in new data, and put the command
void DispCmdTextPosIndex::putdata(int pos1, Displayable *dobj) {
  pos = pos1;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdTextPosIndex::reputdata(int pos1, Displayable *dobj) {
  pos = pos1;
  DispCmd::reput(dobj);
}

//*************************************************************

// define a light ... need the color and position of the light
DispCmdLightDef::DispCmdLightDef(void) 
: DispCmd(DLIGHTDEF, 7 * sizeof(float)) {
  Data = (void *)lightdata;
}

DispCmdLightDef::DispCmdLightDef(int num, float *color, float *pos) 
  : DispCmd(DLIGHTDEF, 7 * sizeof(float)) {
  Data = (void *)lightdata;
  lightdata[0] = (float)num;
  memcpy((void *)(lightdata + 1), (void *)color, 3 * sizeof(float));
  memcpy((void *)(lightdata + 4), (void *)pos, 3 * sizeof(float));
}

// put in new data, and put the command
void DispCmdLightDef::putdata(int num, float *color, float *pos, 
                               Displayable *dobj) {
  lightdata[0] = (float)num;
  memcpy((void *)(lightdata + 1), (void *)color, 3 * sizeof(float));
  memcpy((void *)(lightdata + 4), (void *)pos, 3 * sizeof(float));
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdLightDef::reputdata(int num, float *color, float *pos, 
                                   Displayable *dobj) {
  lightdata[0] = (float)num;
  memcpy((void *)(lightdata + 1), (void *)color, 3 * sizeof(float));
  memcpy((void *)(lightdata + 4), (void *)pos, 3 * sizeof(float));
  DispCmd::reput(dobj);
}

//*************************************************************

// toggle a light on or off
DispCmdLightOnOff::DispCmdLightOnOff(void) 
: DispCmd(DLIGHTONOFF, 2*sizeof(int)) {
  Data = (void *)lightdata;
}

DispCmdLightOnOff::DispCmdLightOnOff(int num, int onoff) 
: DispCmd(DLIGHTONOFF, 2*sizeof(int)) {
  Data = (void *)lightdata;
  lightdata[0] = num;
  lightdata[1] = onoff;
}

// put in new data, and put the command
void DispCmdLightOnOff::putdata(int num, int onoff, Displayable *dobj) {
  lightdata[0] = num;
  lightdata[1] = onoff;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdLightOnOff::reputdata(int num, int onoff, Displayable *dobj) {
  lightdata[0] = num;
  lightdata[1] = onoff;
  DispCmd::reput(dobj);
}

//*************************************************************

// set the current sphere resolution to the given one
DispCmdSphereRes::DispCmdSphereRes(void) : DispCmd(DSPHERERES, sizeof(int)) {
  Data = (void *)(&res);
}

DispCmdSphereRes::DispCmdSphereRes(int newres) 
: DispCmd(DSPHERERES, sizeof(int)) {
  Data = (void *)(&res);
  res = newres;
}

// put in new data, and put the command
void DispCmdSphereRes::putdata(int newres, Displayable *dobj) {
  res = newres;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdSphereRes::reputdata(int newres, Displayable *dobj) {
  res = newres;
  DispCmd::reput(dobj);
}

//*************************************************************

// set the current sphere type to the given one
DispCmdSphereType::DispCmdSphereType(void) 
: DispCmd(DSPHERETYPE, sizeof(int)) {
  Data = (void *)(&type);
}

DispCmdSphereType::DispCmdSphereType(int newtype) 
: DispCmd(DSPHERETYPE, sizeof(int)) {
  Data = (void *)(&type);
  type = newtype;
}

// put in new data, and put the command
void DispCmdSphereType::putdata(int newtype, Displayable *dobj) {
  type = newtype;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdSphereType::reputdata(int newtype, Displayable *dobj) {
  type = newtype;
  DispCmd::reput(dobj);
}

//*************************************************************

// set the current line type to the given one
DispCmdLineType::DispCmdLineType(void) : DispCmd(DLINESTYLE, sizeof(int)) {
  Data = (void *)(&type);
}

DispCmdLineType::DispCmdLineType(int newtype) 
: DispCmd(DLINESTYLE, sizeof(int)) {
  Data = (void *)(&type);
  type = newtype;
}

// put in new data, and put the command
void DispCmdLineType::putdata(int newtype, Displayable *dobj) {
  type = newtype;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdLineType::reputdata(int newtype, Displayable *dobj) {
  type = newtype;
  DispCmd::reput(dobj);
}

//*************************************************************

// set the current line width to the given one
DispCmdLineWidth::DispCmdLineWidth(void) : DispCmd(DLINEWIDTH, sizeof(int)) {
  Data = (void *)(&width);
}

DispCmdLineWidth::DispCmdLineWidth(int newwidth) 
: DispCmd(DLINEWIDTH, sizeof(int)) {
  Data = (void *)(&width);
  width = newwidth;
}

// put in new data, and put the command
void DispCmdLineWidth::putdata(int newwidth, Displayable *dobj) {
  width = newwidth;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdLineWidth::reputdata(int newwidth, Displayable *dobj) {
  width = newwidth;
  DispCmd::reput(dobj);
}

//*************************************************************

// indicate a point which may be picked, and the 'tag' which is associated
// with this point
DispCmdPickPoint::DispCmdPickPoint(void)
: DispCmd(DPICKPOINT, 4*sizeof(float)) {
  Data = (void *)(&postag);
}

DispCmdPickPoint::DispCmdPickPoint(float *pos, int tag) 
: DispCmd(DPICKPOINT, 4*sizeof(float)) {
  Data = (void *)(&postag);
  postag[0] = pos[0]; postag[1] = pos[1]; postag[2] = pos[2];
  postag[3] = (float)tag;
}

// put in new data, and put the command
void DispCmdPickPoint::putdata(float *pos, int tag, Displayable *dobj) {
  postag[0] = pos[0]; postag[1] = pos[1]; postag[2] = pos[2];
  postag[3] = (float)tag;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdPickPoint::reputdata(float *pos, int tag, Displayable *dobj) {
  postag[0] = pos[0]; postag[1] = pos[1]; postag[2] = pos[2];
  postag[3] = (float)tag;
  DispCmd::reput(dobj);
}

//*************************************************************

// indicate a point which may be picked, and the 'tag' which is associated
// with this point.  This uses an indexed position instead of a specific one.
DispCmdPickPointIndex::DispCmdPickPointIndex(void)
: DispCmd(DPICKPOINT_I, 2*sizeof(int)) {
  Data = (void *)(&postag);
}

DispCmdPickPointIndex::DispCmdPickPointIndex(int pos, int tag) 
: DispCmd(DPICKPOINT_I, 2*sizeof(int)) {
  Data = (void *)(&postag);
  postag[0] = pos;
  postag[1] = tag;
}

// put in new data, and put the command
void DispCmdPickPointIndex::putdata(int pos, int tag, Displayable *dobj) {
  postag[0] = pos;
  postag[1] = tag;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdPickPointIndex::reputdata(int pos, int tag, Displayable *dobj) {
  postag[0] = pos;
  postag[1] = tag;
  DispCmd::reput(dobj);
}

//*************************************************************

// indicate a line which may be picked, and the 'tag' which is associated
// with this point
DispCmdPickLine::DispCmdPickLine(void)
: DispCmd(DPICKLINE, 7*sizeof(float)) {
  Data = (void *)(&postag);
}

DispCmdPickLine::DispCmdPickLine(float *pos1, float *pos2, int tag) 
: DispCmd(DPICKLINE, 7*sizeof(float)) {
  Data = (void *)(&postag);
  postag[0] = pos1[0]; postag[1] = pos1[1]; postag[2] = pos1[2];
  postag[3] = pos2[0]; postag[4] = pos2[1]; postag[5] = pos2[2];
  postag[6] = (float)tag;
}

// put in new data, and put the command
void DispCmdPickLine::putdata(float *pos1, float *pos2, int tag,
	Displayable *dobj) {
  postag[0] = pos1[0]; postag[1] = pos1[1]; postag[2] = pos1[2];
  postag[3] = pos2[0]; postag[4] = pos2[1]; postag[5] = pos2[2];
  postag[6] = (float)tag;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdPickLine::reputdata(float *pos1, float *pos2, int tag,
	Displayable *dobj) {
  postag[0] = pos1[0]; postag[1] = pos1[1]; postag[2] = pos1[2];
  postag[3] = pos2[0]; postag[4] = pos2[1]; postag[5] = pos2[2];
  postag[6] = (float)tag;
  DispCmd::reput(dobj);
}

//*************************************************************

// indicate a line which may be picked, and the 'tag' which is associated
// with this point.  This uses an indexed position instead of a specific one.
DispCmdPickLineIndex::DispCmdPickLineIndex(void)
: DispCmd(DPICKLINE_I, 3*sizeof(int)) {
  Data = (void *)(&postag);
}

DispCmdPickLineIndex::DispCmdPickLineIndex(int pos1, int pos2, int tag) 
: DispCmd(DPICKLINE_I, 3*sizeof(int)) {
  Data = (void *)(&postag);
  postag[0] = pos1;
  postag[1] = pos2;
  postag[2] = tag;
}

// put in new data, and put the command
void DispCmdPickLineIndex::putdata(int pos1, int pos2, int tag,
	Displayable *dobj) {
  postag[0] = pos1;
  postag[1] = pos2;
  postag[2] = tag;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdPickLineIndex::reputdata(int pos1, int pos2, int tag,
	Displayable *dobj) {
  postag[0] = pos1;
  postag[1] = pos2;
  postag[2] = tag;
  DispCmd::reput(dobj);
}

//*************************************************************

// indicate a box which may be picked, and the 'tag' which is associated
// with this point.
// This box is specified by giving two opposite corners.
DispCmdPickBox::DispCmdPickBox(void)
: DispCmd(DPICKBOX, 7*sizeof(float)) {
  Data = (void *)(&postag);
}

DispCmdPickBox::DispCmdPickBox(float *pos1, float *pos2, int tag) 
: DispCmd(DPICKBOX, 7*sizeof(float)) {
  Data = (void *)(&postag);
  postag[0] = pos1[0]; postag[1] = pos1[1]; postag[2] = pos1[2];
  postag[3] = pos2[0]; postag[4] = pos2[1]; postag[5] = pos2[2];
  postag[6] = (float)tag;
}

// put in new data, and put the command
void DispCmdPickBox::putdata(float *pos1, float *pos2, int tag,
	Displayable *dobj) {
  postag[0] = pos1[0]; postag[1] = pos1[1]; postag[2] = pos1[2];
  postag[3] = pos2[0]; postag[4] = pos2[1]; postag[5] = pos2[2];
  postag[6] = (float)tag;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdPickBox::reputdata(float *pos1, float *pos2, int tag,
	Displayable *dobj) {
  postag[0] = pos1[0]; postag[1] = pos1[1]; postag[2] = pos1[2];
  postag[3] = pos2[0]; postag[4] = pos2[1]; postag[5] = pos2[2];
  postag[6] = (float)tag;
  DispCmd::reput(dobj);
}

//*************************************************************

// indicate a box which may be picked, and the 'tag' which is associated
// with this point.  This uses an indexed position instead of a specific one.
// This box is specified by giving two opposite corners.
DispCmdPickBoxIndex::DispCmdPickBoxIndex(void)
: DispCmd(DPICKBOX_I, 3*sizeof(int)) {
  Data = (void *)(&postag);
}

DispCmdPickBoxIndex::DispCmdPickBoxIndex(int pos1, int pos2, int tag) 
: DispCmd(DPICKBOX_I, 3*sizeof(int)) {
  Data = (void *)(&postag);
  postag[0] = pos1;
  postag[1] = pos2;
  postag[2] = tag;
}

// put in new data, and put the command
void DispCmdPickBoxIndex::putdata(int pos1, int pos2, int tag,
	Displayable *dobj) {
  postag[0] = pos1;
  postag[1] = pos2;
  postag[2] = tag;
  DispCmd::put(dobj);
}

// put in new data, and reput the command
void DispCmdPickBoxIndex::reputdata(int pos1, int pos2, int tag,
	Displayable *dobj) {
  postag[0] = pos1;
  postag[1] = pos2;
  postag[2] = tag;
  DispCmd::reput(dobj);
}

