/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: DrawMolItem.C,v $
 *	$Author: dalke $	$Locker:  $		$State: Exp $
 *	$Revision: 1.38 $	$Date: 1995/06/02 20:41:03 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Child Displayable component of a molecule; this is responsible for doing
 * the actual drawing of a molecule.  It contains an atom color, atom
 * selection, and atom representation object to specify how this component
 * should look.
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>		// for systems()
#include <unistd.h>
#include <sys/types.h>
#include <math.h>

#include "DrawMolItem.h"
#include "DrawMolecule.h"
#include "DispCmds.h"
#include "Inform.h"
#include "Scene.h"
#include "CoorPDB.h"		// for write_pdb_record

#ifdef VMDALPHA_SHAPE
// This is not yet available outside of the Theoretical Biophysics
// Group and is still under development.
#include "AlphaShape.h"
#endif

// draw commands used by this object to indicate points to pick
static DispCmdPickPoint pickPoint;
static DispCmdPickPointIndex pickPointIndex;

//////////////////////////  constructor  

DrawMolItem::DrawMolItem(DrawMolecule *dm, AtomColor *ac, AtomRep *ar,
	AtomSel *as) 
	: Displayable3D(MULT, dm->name, dm, dm->nAtoms/8 + 2),
	  cmdTextX(".") {

  MSGDEBUG(1,"Creating new molecule representation, for parent molecule:\n");
  MSGDEBUG(1," ==> '" << dm->name << "'" << sendmsg);

  // save data and pointers to drawing method objects
  mol = dm;
  atomColor = ac;
  atomRep = ar;
  atomSel = as;
  
  // create some useful display commands
  dataCmd = new DispCmdData(3 * mol->nAtoms, NULL);

  // signal that we need a complete refresh next prepare
  needRegenerate = FORCE_REGEN;
  needNewFrame = TRUE;

  // register with the pick list, to indicate we have objects that may
  // be selected with the mouse
  register_with_picklist(this->origScene);
}

//////////////////////////  destructor  
DrawMolItem::~DrawMolItem(void) {

  MSGDEBUG(2,"Deleting DrawMolItem '" << name << "' ..." << sendmsg);

  delete atomColor;
  delete atomRep;
  delete atomSel;

  delete dataCmd;
}


//////////////////////////////// private routines 

// regenerate the command list
void DrawMolItem::create_cmdlist(void) {
  float *framepos;

  // do we need to recreate everything?
  if(needRegenerate) {
  
    MSGDEBUG(2, "Regenerating display list for molecule '"<< name<< "' ...");
    MSGDEBUG(2, sendmsg);

    // regenerate both data block and display commands
    needNewFrame = FALSE;
    reset_disp_list();

    // only put in commands if there is a current frame
    if(mol->frame() >= 0) {
      // get current position coordinates
      framepos = (mol->current())->pos;

      // first put in data command
      dataCmd->putdata(framepos, this);
      
      // now put in drawing commands, which index into the above block
      if(atomRep->method() == AtomRep::LINES)
      draw_lines(framepos);
      else if(atomRep->method() == AtomRep::BONDS)
        draw_bonds(framepos);
      else if(atomRep->method() == AtomRep::CPK)
        draw_cpk(framepos);
      else if(atomRep->method() == AtomRep::POINTS)
        draw_points(framepos);
      else if(atomRep->method() == AtomRep::VDW)
        draw_vdw();
      else if(atomRep->method() == AtomRep::DOTTED)
        draw_dotted();
      else if(atomRep->method() == AtomRep::LICORICE)
        draw_licorice(framepos);
      else if(atomRep->method() == AtomRep::RIBBONS)
        draw_ribbons(framepos);
      else if (atomRep->method() == AtomRep::TUBE)
        draw_tube(framepos);
#ifdef VMDALPHA_SHAPE
      else if (atomRep->method() == AtomRep::ALPHASHAPE)
        draw_alphaShape(framepos);
#endif
      else if(atomRep->method() == AtomRep::REPOFF) {
       	// don't do anything ... draws a blank object
      } else
        msgErr << "Illegal atom representation in DrawMolecule." << sendmsg;
    }

  } else if(needNewFrame) {
    // just update the coordinates ... assume there is a current frame
    needNewFrame = FALSE;
    dataCmd->reputdata((mol->current())->pos, this);
  }
  // and, finally, I don't need a new regeneration
  needRegenerate = NO_REGEN;
}

//////////////////////////////// protected virtual routines
// do action due to the fact that a color for the given ColorList for
// the specified category has changed
void DrawMolItem::do_color_changed(ColorList *collist, int ccat) {

  // check the category ... see if is for the current coloring method
  if(atomColor && atomColor->current_color_use(collist, ccat))
    change_color(atomColor);		// recalcs colors; signals need redraw
}


//////////////////////////////// public routines 
// prepare for drawing ... do any updates needed right before draw.
void DrawMolItem::prepare(DisplayDevice *) {

  // see if the frame has changed
  if(mol->has_frame_changed()) {
    needNewFrame = TRUE;
    needRegenerate |= FORCE_REGEN;
  }
  
  create_cmdlist();
}


// change the atom coloring method.  Return success.
int DrawMolItem::change_color(AtomColor *ac) {
  if(ac) {
    *atomColor = *ac;
    needRegenerate |= COL_REGEN;
    return TRUE;
  } else
    return FALSE;
}


// change the atom rep method.  Return success.
int DrawMolItem::change_rep(AtomRep *ar) {
  if(ar) {
     *atomRep = *ar;
     needRegenerate |= REP_REGEN;
     return TRUE;
  } else {
     return FALSE;
  }
}


// change the atom selection method.  Return success.
int DrawMolItem::change_sel(AtomSel *as) {
  if(as) {
    *atomSel = *as;
    needRegenerate |= SEL_REGEN;
    return TRUE;
  } else
    return FALSE;
}


// return whether the Nth atom is displayed.
int DrawMolItem::atom_displayed(int n) {
  return (atomSel != NULL && mol != NULL && n >= 0 && n < mol->nAtoms &&
  		 atomSel->on[n]);
}


//////////////////////////////// drawing rep routines
void DrawMolItem::draw_lines(float *framepos) {
  float pos2[3], *fp1, *fp2;
  Atom *a1, *a2;
  int i,j,k,a2n,bondsdrawn;
  
  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms with lines and points...");
  MSGDEBUG(2, sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn off material characteristics, set line style
  cmdMaterials.putdata(FALSE,this);
  cmdLineType.putdata(SOLIDLINE, this);
  cmdLineWidth.putdata(atomRep->line_thickness(), this);

  // check all atoms if they are displayed, and if so draw bonds as lines
  for(i=0; i < mol->nAtoms; i++) {
    // for each atom, draw half-bond to other displayed atoms
    if(atomSel->on[i]) {
      a1 = mol->atom(i);
      bondsdrawn = 0;

      // set color
      cmdColorIndex.putdata(atomColor->color[i], this);
      fp1 = framepos + 3*i;
      
      // draw half-bond to each bonded, displayed partner
      // NOTE: bond mid-points are calculated TWICE, once for each atom.
      // Why? So each atom is drawn entirely, including all it's half-bonds,
      // after a single set-color command, instead of having a set-color
      // command for each bond.  This reduces total drawing primitives
      // considerably, at the cost of extra calculation here.
      for(j=0; j < a1->bonds; j++) {
        a2n = a1->bondTo[j];
        if(atomSel->on[a2n]) {	// bonded atom displayed?
	  fp2 = framepos + 3*a2n;
          a2 = mol->atom(a2n);
	  for(k=0; k < 3; k++)		// calc midpoint
	    pos2[k] = 0.5 * ( *(fp1 + k) + *(fp2 + k) );
	  cmdLine.putdata(fp1, pos2, this);	// draw line
	  bondsdrawn++;			// increment counter of bonds to atom i
	}
      }
      
      // if this atom is all alone (no bonds drawn), make an 'x'
      if(!bondsdrawn) {
        cmdTextPosIndex.putdata(3*i, this);
	cmdTextX.put(this);
      }

      // indicate this atom can be picked
      pickPointIndex.putdata(3*i, i, this);
    }
  }
}


// draw all atoms just as spheres of vdw radius
void DrawMolItem::draw_just_spheres(void) {
  float radscale;
  int i;

  radscale = atomRep->sphere_rad();

  // check all atoms if they are displayed, and if so draw a sphere
  for(i=0; i < mol->nAtoms; i++) {
    if(atomSel->on[i]) {
      // set color
      cmdColorIndex.putdata(atomColor->color[i], this);
      
      // draw sphere
      cmdSphereIndex.putdata(3*i, (mol->atom(i))->radius() * radscale, this);

      // indicate this atom can be picked
      pickPointIndex.putdata(3*i, i, this);
    }
  }
}


// draws each atom as a dot
void DrawMolItem::draw_points(float *framepos) {
   DispCmdPoint cmdPoint;
   cmdMaterials.putdata(FALSE,this);

   // check all atoms if they are displayed, and if so, plot a point
   for (int i=mol->nAtoms-1; i>=0; i--) {
      if (atomSel->on[i]) {
	 // set color
	 cmdColorIndex.putdata(atomColor->color[i], this);

	 // draw the point
	 cmdPoint.putdata(framepos + 3*i, this);

	 // and make this pickable
	 pickPointIndex.putdata(3*i, i, this);
      }
   }
}

// draws molecule as solid, lighted VDW spheres
void DrawMolItem::draw_vdw(void) {
  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms as solid VDW spheres...");
  MSGDEBUG(2, sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn on material characteristics
  cmdMaterials.putdata(TRUE,this);

  // set sphere mode and resolution
  cmdSphres.putdata(atomRep->sphere_res(), this);
  cmdSphtype.putdata(SOLIDSPHERE, this);
  
  // draw all atoms as spheres
  draw_just_spheres();

}


// draws molecule as dotted VDW spheres
void DrawMolItem::draw_dotted(void) {

  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms as dotted VDW spheres...");
  MSGDEBUG(2,sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn off material characteristics
  cmdMaterials.putdata(FALSE,this);

  // set sphere mode and resolution
  cmdSphres.putdata(atomRep->sphere_res(), this);
  cmdSphtype.putdata(POINTSPHERE, this);
  
  // draw all atoms as spheres
  draw_just_spheres();
}


void DrawMolItem::draw_cpk(float *framepos) {
  float pos2[3], *fp1, *fp2;
  Atom *a1, *a2;
  int i,j,k,a2n,bondsdrawn;
  float radscale = 0.25 * atomRep->sphere_rad();
  float brad = 0.25 * atomRep->bond_rad();
  int bres = atomRep->bond_res();

  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms with CPK method ...");
  MSGDEBUG(2,sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn on material characteristics
  cmdMaterials.putdata(TRUE,this);
  
  // set sphere and cylinder mode and resolution
  cmdSphres.putdata(atomRep->sphere_res(), this);
  cmdSphtype.putdata(SOLIDSPHERE, this);

  // check all atoms if they are displayed, and if so draw bonds as cylinders,
  // and the atom as a sphere, with radius = 1/4 vdw radius
  for(i=0; i < mol->nAtoms; i++) {
    // for each atom, draw half-bond to other displayed atoms
    if(atomSel->on[i]) {
      a1 = mol->atom(i);
      bondsdrawn = 0;

      // set color
      cmdColorIndex.putdata(atomColor->color[i], this);
      fp1 = framepos + 3*i;
      
      // draw a sphere for the atom
      cmdSphereIndex.putdata(3*i, a1->radius() * radscale, this);
      
      // draw half-bond to each bonded, displayed partner
      for(j=0; j < a1->bonds; j++) {
        a2n = a1->bondTo[j];
        if(atomSel->on[a2n]) {	// bonded atom displayed?
	  fp2 = framepos + 3*a2n;
          a2 = mol->atom(a2n);
	  for(k=0; k < 3; k++)		// calc midpoint
	    pos2[k] = 0.5 * ( *(fp1 + k) + *(fp2 + k) );
	  if(i < a2n)
	    cmdCylinder.putdata(fp1, pos2, brad, bres, this);	// draw cyl
	  else
	    cmdCylinder.putdata(pos2, fp1, brad, bres, this);
	  bondsdrawn++;			// increment counter of bonds to atom i
	}
      }

      // indicate this atom can be picked
      pickPointIndex.putdata(3*i, i, this);
    }
  }
}


void DrawMolItem::draw_licorice(float *framepos) {
  float pos2[3], *fp1, *fp2;
  Atom *a1, *a2;
  int i,j,k,a2n,bondsdrawn;
  float brad = atomRep->bond_rad();
  int bres = atomRep->bond_res();

  MSGDEBUG(2,"  Drawing " << mol->nAtoms << " atoms with licorice bonds ...");
  MSGDEBUG(2,sendmsg);

  // set up general drawing characteristics for this drawing method
  // turn on material characteristics
  cmdMaterials.putdata(TRUE,this);
  
  // set sphere and cylinder mode and resolution
  cmdSphres.putdata(atomRep->sphere_res(), this);
  cmdSphtype.putdata(SOLIDSPHERE, this);

  // check all atoms if they are displayed, and if so draw bonds as cylinders,
  // and the atom as a sphere
  for(i=0; i < mol->nAtoms; i++) {
    // for each atom, draw half-bond to other displayed atoms
    if(atomSel->on[i]) {
      a1 = mol->atom(i);
      bondsdrawn = 0;

      // set color
      cmdColorIndex.putdata(atomColor->color[i], this);
      fp1 = framepos + 3*i;
      
      // draw a sphere for the atom, with same radius as bond
      cmdSphereIndex.putdata(3*i, brad, this);
      
      // draw half-bond to each bonded, displayed partner
      for(j=0; j < a1->bonds; j++) {
        a2n = a1->bondTo[j];
        if(atomSel->on[a2n]) {	// bonded atom displayed?
	  fp2 = framepos + 3*a2n;
          a2 = mol->atom(a2n);
	  for(k=0; k < 3; k++)		// calc midpoint
	    pos2[k] = 0.5 * ( *(fp1 + k) + *(fp2 + k) );
	  if(i < a2n)
	    cmdCylinder.putdata(fp1, pos2, brad, bres, this);	// draw cyl
	  else
	    cmdCylinder.putdata(pos2, fp1, brad, bres, this);
	  bondsdrawn++;			// increment counter of bonds to atom i
	}
      }

      // indicate this atom can be picked
      pickPointIndex.putdata(3*i, i, this);
    }
  }
}


//   This draws a cylindrical bond from atom 'i' to atom 'k', but
// only if both are selected.  Each end of the cylinder is colored
// appropriately.  If there are bonds connected to the ends of this
/// i-k bond, then the i-k bond is extended so that the bonds have
// no gap between them
void DrawMolItem::draw_bonds(float *framepos)
{
   Atom *a1, *a2;
   int g, h, i, j, k, l, m;
   float brad = atomRep->bond_rad();
   int bres = atomRep->bond_res();
   
   // set up general drawing characteristics for this drawing method
   // turn on material characteristics
   cmdMaterials.putdata(TRUE,this);
   
   // set sphere and cylinder mode and resolution
   cmdSphres.putdata(atomRep->sphere_res(), this);
   cmdSphtype.putdata(SOLIDSPHERE, this);
   
   for(i=0; i < mol->nAtoms; i++) {
      // find a selected atom
      if (atomSel->on[i]) {
	 a1 = mol->atom(i);
	 // of the bonded atoms with index > i,  (See the comment
	 // later on about picking 'k' if you don't want to use the
	 // '>')
	 for (j=0; j<a1->bonds; j++) {
	    int set_pickable = FALSE;
	    // find one that is turned on
	    k = a1 -> bondTo[j];
	    if (k > i && atomSel->on[k]) {
	       a2 = mol->atom(k);
 // To make things look nice, I look for a bond that extends each end
 // of the bond.  So I look for an atom 'm' connected to 'k' which
 // is on and is not 'i'.  I also look for an atom 'g' connected to
 // 'i' which is not 'k'.
 // I deliberatly choose to look only at 'on' atoms.  I think it is
 // a good idea, but I guess you are the judge
	       for (l=a2->bonds-1; l>=0; l--) {
		  m = a2 -> bondTo[l];
		  if (m != i  && atomSel->on[m]) {
		     break;
		  }
	       } // (m is good, or l==-1)

	       for (h=a1->bonds-1; h>=0; h--) {
		  g = a1 -> bondTo[h];
		  if (g != k && atomSel->on[g]) {
		     break;
		  }
	       } // g is good, or h==-1)

	       // at this point I have:
	       // g = {good or -1}, i, k, and m = {good or -1}
	       // so I draw the appropriate cylinder
	       float *p1, *p2, p3[3], *p4, *p5;
	       if (h == -1) p1 = NULL; else p1 = framepos + 3*g;
	       p2 = framepos + 3*i;
	       p4 = framepos + 3*k;
	       if (l==-1) p5 = NULL; else p5 = framepos + 3*m;
	       // find the bisector
	       add(p3, p2, p4);
	       p3[0] /= 2.0;
	       p3[1] /= 2.0;
	       p3[2] /= 2.0;
	       // now I can draw the bonds
	       cmdColorIndex.putdata(atomColor->color[i], this);
	       make_connection(p1, p2, p3, p4, brad, bres, 1);
	       cmdColorIndex.putdata(atomColor->color[k], this);
	       make_connection(p2, p3, p4, p5, brad, bres, 1);
	       // and make it pickable (once!)
	       if (!set_pickable) {
		  set_pickable = !set_pickable;
		  pickPointIndex.putdata(3*i, i, this);
		  // this is tricky.  If 'k' is on, but the only
		  // atom bonded to it is 'i', then 'k' won't be
		  // visited.  If that is the case (ie, p5 is NULL)
		  // then I have to mark 'k' as picked here instead
		  // of later
		  if (!p5) {
		     pickPointIndex.putdata(3*k, k, this);
		  }
	       }
		  
	    }  // found k atom
	 } // searching along i's bonds
      } // found i
   } // searching each atom
}


static void make_spline_Q_matrix(float q[4][3], float basis[4][4],
				 float *pts)
{
   register i, j, dim;
   for (i = 0; i<4; i++) {
      q[i][0] = q[i][1] = q[i][2] = 0;
      for (j = 0; j<4; j++) {
	 for (dim = 0; dim<3; dim++) {
	    q[i][dim] += basis[i][j]*pts[j*3+dim];
	 }
      }
   }
}

static void make_spline_interpolation(float out[3], register float w,
				      float q[4][3])
{
   out[0] = w * (w * (w * q[0][0] + q[1][0]) + q[2][0]) + q[3][0];
   out[1] = w * (w * (w * q[0][1] + q[1][1]) + q[2][1]) + q[3][1];
   out[2] = w * (w * (w * q[0][2] + q[1][2]) + q[2][2]) + q[3][2];
}


/*
static void shift_matrix(float mat[4][3], float v[3])
{
   mat[0][0] = mat[1][0];  mat[1][0] = mat[2][0];
     mat[2][0] = mat[3][0];  mat[3][0] = v[0];
   mat[0][1] = mat[1][1];  mat[1][1] = mat[2][1];
     mat[2][1] = mat[3][1];  mat[3][1] = v[1];
   mat[0][2] = mat[1][2];  mat[1][2] = mat[2][2];
     mat[2][2] = mat[3][2];  mat[3][2] = v[2];
}
*/

// given the current cylinder and the previous and next cylinders,
// find and draw the cylinder (or line) that best fits without making
// an overlap.  prev can be NULL, in which case it acts as if the
// previous cylinder was linear with the one to draw.  Ditto (but
// opposite) for next.
//
//  The problem is that if I make two cylinders touch head to tail,
// but at an angle, there is a gap between the two.  I can get
// around it by extending the cylinders appropriately.  
void DrawMolItem::make_connection(float *prev, float *start, float *end,
				  float *next, float rad, int res, int use_cyl)
{
   if (!start || !end) {
      msgErr << "Trying to make an extended cylinder with NULL end point(s)"
	     << sendmsg;
      return;
   }
   if (!use_cyl) {
      cmdLine.putdata(start, end, this);
      return;
   }
   // now for a bit of vector math
   float new_start[3], new_end[3];
   float temp[6][3];
   float length;

   if (!prev) {          // along the previous cylinder (a)
      subtract(temp[0], end, start);  // when you don't know a
   } else {
      subtract(temp[0], start, prev);
   }
   subtract(temp[1], end, start);   // this cylinder (b)
   cross_prod(temp[2], temp[0], temp[1]);  // perp to both a and b
   // does this exceed threshhold to start extension (might speed things up)
   if (dot_prod(temp[2], temp[2]) > 0.0001/rad) {
      cross_prod(temp[3], temp[0], temp[2]);  // perp to a and away from b
      normalize(temp[3]);
      cross_prod(temp[4], temp[1], temp[2]);  // perp to b and away from a
      normalize(temp[4]);
      add(temp[5], temp[3], temp[4]);     // farthest intersection of a and b
      // find the intersection pt
      temp[5][0] *= rad/2.0; temp[5][1] *= rad/2.0; temp[5][2] *= rad/2.0;
      normalize(temp[1]);                     // unit vector along b
      length = dot_prod(temp[1], temp[5]);
   } else {
      length = 0;
   }
   new_start[0] = start[0] + length * temp[1][0];  // and the new starting
   new_start[1] = start[1] + length * temp[1][1];  // point
   new_start[2] = start[2] + length * temp[1][2];

   // this is a copy of the previous except that I swapped the
   // direction of travel.  Now a points backwards along then "next" cylinder
   // and b is this cylinder.
   if (!next) {           // backwards along the previous cylinder (a)
      subtract(temp[0], start, end);  // when you don't know a
   } else {
      subtract(temp[0], end, next);
   }
   subtract(temp[1], start, end);   // this cylinder (b)
   cross_prod(temp[2], temp[0], temp[1]);  // perp to both a and b
   // does this exceed threshhold to start extension (might speed things up)
   if (dot_prod(temp[2], temp[2]) > 0.0001/rad) {
      cross_prod(temp[3], temp[0], temp[2]);  // perp to a and away from b
      normalize(temp[3]);
      cross_prod(temp[4], temp[1], temp[2]);  // perp to b and away from a
      normalize(temp[4]);
      add(temp[5], temp[3], temp[4]);     // farthest intersection of a and b
      // find the intersection pt
      temp[5][0] *= rad/2.0; temp[5][1] *= rad/2.0; temp[5][2] *= rad/2.0;
      normalize(temp[1]);                     // unit vector along b
      length = dot_prod(temp[1], temp[5]);
   } else {
      length = 0;
   }
   new_end[0] = end[0] + length * temp[1][0];  // and the new starting
   new_end[1] = end[1] + length * temp[1][1];  // point
   new_end[2] = end[2] + length * temp[1][2];
   cmdCylinder.putdata(new_start, new_end, rad, res, this);
}

// These are different ways to define a general cubic spline
// see Foley and Van Dam, et. al., Computer Graphics, p505 or so

// this one was too sharply curved for my tastes
//   float CatmullRom_basis[4][4]={{-1.0/2.0,  3.0/2.0, -3.0/2.0,  1.0/2.0},
//				 { 2.0/2.0, -5.0/2.0,  4.0/2.0, -1.0/2.0},
//				 {-1.0/2.0,  0.0/2.0,  1.0/2.0,  0.0/2.0},
//				 { 0.0/2.0,  2.0/2.0,  0.0/2.0,  0.0/2.0}};

// this define make the next basis identical to CatmullRom
//   #define SLOPE 2.0
// This deemphasizes the slope and makes things look nicer (IMHO)
#define SLOPE 1.25
   float modified_CR_basis[4][4] = {
      {-1.0/SLOPE,  -1.0/SLOPE + 2.0,  1.0/SLOPE - 2.0,  1.0/SLOPE},
      { 2.0/SLOPE,   1.0/SLOPE - 3.0,   -2.0/SLOPE+3.0, -1.0/SLOPE},
      {-1.0/SLOPE,                 0,        1.0/SLOPE,          0},
      {         0,               1.0,                0,          0}
   };
   // This doesn't describe the system very nicely as the lines don't
   // go through the control points (which are the C-alphas)
//   float Bspline_basis[4][4]={{-1.0/6.0,  3.0/6.0, -3.0/6.0,  1.0/6.0},
//			      { 3.0/6.0, -6.0/6.0,  3.0/6.0,  0.0/6.0},
//			      {-3.0/6.0,  0.0/6.0,  3.0/6.0,  0.0/6.0},
//			      { 1.0/6.0,  4.0/6.0,  1.0/6.0,  0.0/6.0}};


// this fctn does the actual work of making the spline curve from
// a set of coordinates.  There are num points, coords goes from
// -2 to num+1 and has 3 floats per coord, idx also goes from
// -2 to num+1 and has either the atom number or -1.  The rest
// specify if cylinders should be used and (if so) the radius and
// resolution
void DrawMolItem::draw_spline_curve(int num, float *coords, int *idx,
				    int use_cyl, float b_rad, int b_res)
{
   float q[4][3];
   float prev[2][3];  // the last couple points for the previous spline
   float final[7][3]; // the points of the current approzimation
   int last_loop = -10;
   int loop;
   for (loop=-1; loop<num; loop++) { // go through the array
      
      // check if I'll need to do any computations
      // this make things faster, but its a bit more of a headache
      // later on
      if (idx[loop] >=0 && atomSel->on[idx[loop]] ||
	  idx[loop+1] >=0 && atomSel->on[idx[loop+1]]) {
	 // make the Q matrix
	 make_spline_Q_matrix(q, modified_CR_basis, coords+(loop-1)*3);
	 
	 // evaluate the interpolation between atom "loop" and
	 // "loop+1".  I'll make this in 6 pieces
	 make_spline_interpolation(final[0], 0.0/6.0, q);
	 make_spline_interpolation(final[1], 1.0/6.0, q);
	 make_spline_interpolation(final[2], 2.0/6.0, q);
	 make_spline_interpolation(final[3], 3.0/6.0, q);
	 make_spline_interpolation(final[4], 4.0/6.0, q);
	 make_spline_interpolation(final[5], 5.0/6.0, q);
	 make_spline_interpolation(final[6], 6.0/6.0, q);
	 
	 // draw what I need for atom 'loop'
	 if (idx[loop] >= 0 &&          // this is a real atom
	     atomSel->on[idx[loop]]) {  // and it it turned on
	    
	    // here is the trickiness I was talking about
	    // I need to see if there was a computation for
	    // the first part of this atom
	    if (last_loop != loop - 1) {  // no there wasn't
	       cmdColorIndex.putdata(atomColor->color[idx[loop]], this);
	       make_connection( NULL, final[0], final[1], final[2],
				b_rad, b_res, use_cyl);
	    } else {
       // finish up drawing the previous part of this residue (I couldn't
       // do this before since I didn't know how to angle the final cylinder
       // to get a nice match up to the first of the current cylinders)
	       make_connection( prev[0], prev[1], final[0], final[1],
				b_rad, b_res, use_cyl);
	       make_connection( prev[1], final[0], final[1], final[2],
				b_rad, b_res, use_cyl);
	    }
	    make_connection(final[0], final[1], final[2], final[3],
			    b_rad, b_res, use_cyl);
	    make_connection(final[1], final[2], final[3], final[4],
			    b_rad, b_res, use_cyl);
	    // indicate this atom can be picked
	    pickPointIndex.putdata(3*idx[loop], idx[loop], this);
	 }
	 
	 // draw what I can for atom 'loop+1'
	 if (idx[loop+1] >= 0 &&
	     atomSel->on[idx[loop+1]]) {
	    cmdColorIndex.putdata(atomColor->color[idx[loop+1]], this);
	    make_connection(final[2], final[3], final[4], final[5],
			    b_rad, b_res, use_cyl);
	    make_connection(final[3], final[4], final[5], final[6],
			    b_rad, b_res, use_cyl);
	    last_loop = loop;
	 }
	 copy(prev[0], final[4]);  // save for the
	 copy(prev[1], final[5]);  // next interation
      } /// else nothing to draw
   } // gone down the fragment
}

// draw a spline ribbon.  The math is the same as a tube. 
// However, two splines are calculated, one for coords+perp
// and the other along coords-perp.  Triangles are used to
// connect the curves.  If cylinders are used,  they are
// drawn along the spline paths.  The final result should
// look something like:
//    ooooooooooooooooooo   where the oooo are the (optional) cylinders
//    ! /| /| /| /| /| /!   the ! are a control point, in the ribbon center
//    |/ |/ |/ |/ |/ |/ |   the /| are the triangles  (size per interpolation)
//    ooooooooooooooooooo   the edges go through coords[i] +/- offset[i]

void DrawMolItem::draw_spline_ribbon(int num, float *coords,
   float *offset, int *idx, int use_cyl, float b_rad, int b_res)
{
   float q[4][3];
   // for the next 2 variables, the information for this residue
   // starts at element 2; elements 0 and 1 are copies of the last
   // two elements of the previous residue
   float pts[2][9][3];  // data for 2 edges, 7 points (+2 overlaps from
                        // the previous interpolation) and x,y,z
   
   //  contains the norms as [edge][interpolation point][x/y/z]
   float norms[2][8][3]; 
   // contains the summed norms of the sequential triangles
   // the first element (# 0) contains the info for what should have
   // been the last two triangles of the previous residue.  In other
   // words, tri_norm[0][0] contains the summed norm for the point
   // located at pts[1][0]
   float tri_norm[2][7][3]; 

   int last_loop = -10;
   int loop, i;
   float *tmp_ptr = (float *) malloc(2*(num+4) * sizeof(float) * 3);
   if (!tmp_ptr) {
      msgErr << "Cannot make a ribbon; not enough memory!" << sendmsg;
      return;
   }
   float *edge[2];

   // copy the coordinates +/- the offsets into the temp ("edge") arrays
   edge[0] = tmp_ptr + 2*3;
   memcpy(edge[0]-2*3, coords-2*3, (num+4)*sizeof(float)*3);
   edge[1] = edge[0] + (num+4)*3;
   memcpy(edge[1]-2*3, coords-2*3, (num+4)*sizeof(float)*3);
   for (int j=-2*3; j<(num+2)*3-1; j++) {
      edge[0][j] += offset[j];
      edge[1][j] -= offset[j];
   }

   // go through the data points
   for (loop=-1; loop<num; loop++) {
      // If I'm to draw anything....
      if (idx[loop] >= 0 && atomSel->on[idx[loop]] ||
	  idx[loop+1] >= 0 && atomSel->on[idx[loop+1]] ) {

	 // construct the interpolation points (into the "pts" array)
	 // remember, these are offset by two to keep some information
	 // about the previous residue in the first 2 elements
	 for (i=0; i<=1; i++) {
	    make_spline_Q_matrix(q, modified_CR_basis, edge[i]+(loop-1)*3);
	    make_spline_interpolation(pts[i][2], 0.0/6.0, q);
	    make_spline_interpolation(pts[i][3], 1.0/6.0, q);
	    make_spline_interpolation(pts[i][4], 2.0/6.0, q);
	    make_spline_interpolation(pts[i][5], 3.0/6.0, q);
	    make_spline_interpolation(pts[i][6], 4.0/6.0, q);
	    make_spline_interpolation(pts[i][7], 5.0/6.0, q);
	    make_spline_interpolation(pts[i][8], 6.0/6.0, q);
	 }
	 // make the normals for each new point.
	 for (i=2; i<8; i++) {
	    float diff1[3], diff2[3];
	    subtract(diff1, pts[1][i+1], pts[1][i+0]);
	    subtract(diff2, pts[1][i+1], pts[0][i+0]);
	    cross_prod(norms[1][i], diff1, diff2);
	    subtract(diff1, pts[0][i+1], pts[0][i+0]);
	    cross_prod(norms[0][i], diff1, diff2);
	 }
	 //   if this wasn't a continuation, I need to set the
	 // first 2 elements properly so the norms are smooth
	 if (last_loop != loop-1) {
	    copy(norms[0][0], norms[0][2]);
	    copy(norms[1][0], norms[1][2]);
	    copy(norms[0][1], norms[0][2]);
	    copy(norms[1][1], norms[1][2]);
	 }
	 // sum up the values of neighboring triangles to make
	 // a smooth normal
	 for (i=0; i<8-1; i++) {
	    for (int j=0; j<=1; j++) {
	       add(tri_norm[j][i], norms[j][i+1],     // "this" triangle
		                   norms[1-j][i+1]);  // opposite strand
	       add(tri_norm[j][i], tri_norm[j][i],
		                   norms[j][i]);      // prev on strand
	    }
	 }
	 
	 
	 // draw what I need for atom 'loop'
	 if (idx[loop] >= 0 &&          // this is a real atom
	     atomSel->on[idx[loop]]) {  // and it it turned on
	    if (last_loop != loop - 1) {  // do prev. points exist? if not
	       // then I don't know if the color was properly set, so
	       // I'll set it here
	       cmdColorIndex.putdata(atomColor->color[idx[loop]], this);
	    }
	    // draw the cylinders to finish off the last residue, if
	    // need be, and draw the ones for this half of the residue
	    // The cylinders are drawn on the top and bottom of the
	    // residues
	    if (use_cyl) { // draw top/bot edge cylinders if need be

	       // Special case the first cylinder because I want
	       // it to be a smooth continuation from the previous
	       // cylinder; assuming it exists
	       if (last_loop != loop-1) {  // continue from previous?
		  for (int i=0; i<=1; i++) {  // doesn't exist, so
		     make_connection( NULL, pts[i][2], pts[i][3],
				      pts[i][4], b_rad, b_res, use_cyl);
		  }
	       } else { // there was a previous cylinder, so be smooth
		  for (i=0; i<=1; i++) {
	  make_connection(pts[i][0], pts[i][1], pts[i][2], pts[i][3],
			  b_rad, b_res, use_cyl);
	  make_connection(pts[i][1], pts[i][2], pts[i][3], pts[i][4],
			  b_rad, b_res, use_cyl);
                  }
	       }
	       
	       // and draw the rest of the cylinders for this 1/2 residue
	       for (i=0; i<=1; i++) {
	  make_connection( pts[i][2], pts[i][3], pts[i][4], pts[i][5],
			   b_rad, b_res, use_cyl);
	  make_connection( pts[i][3], pts[i][4], pts[i][5], pts[i][6],
			   b_rad, b_res, use_cyl);
               }
	    } // drew cylinders

	    // At this point I draw the triangles that connect the cylinders
	    // and which make up the ribbon proper
	    // the funky start condition is so that I start at
	    // pts[][1] if I need to finish the previous residue, or
	    // pts[][2] if I don't
	    for (i= (last_loop == loop-1 ? 1: 2); i<5; i++) {
       cmdTriangle.putdata(pts[1][1+i], pts[1][0+i], pts[0][0+i],
			   tri_norm[1][0+i], tri_norm[1][-1+i],
			   tri_norm[0][-1+i], this);
       cmdTriangle.putdata(pts[0][1+i], pts[1][1+i], pts[0][0+i],
			   tri_norm[0][0+i], tri_norm[1][0+i],
			   tri_norm[0][-1+i], this);

            // indicate this atom can be picked
            // this spot is in the middle of the ribbon, both in length
            // and in width
	    pickPointIndex.putdata(3*idx[loop], idx[loop], this);
            }
	 }  // atom 'loop' finished
	 
	 // draw what I can for atom 'loop+1'
	 if (idx[loop+1] >= 0 &&
	     atomSel->on[idx[loop+1]]) {
	    // If this is on, then I may have to change the color,
	    // since I'm lazy, I won't check to see if I _have_ to change it
	    // but assume I do
	    cmdColorIndex.putdata(atomColor->color[idx[loop+1]], this);
	    // I can draw the first couple of cylinders, but I need knowledge
	    // of what goes on next to get a smooth fit.  Thus, I won't
	    // draw them here.
	    // I can't draw the last two cylinders.
	    if (use_cyl) {
	       for (i=0; i<=1; i++) {
		  make_connection(pts[i][4], pts[i][5], pts[i][6], pts[i][7],
				  b_rad, b_res, use_cyl);
		  make_connection(pts[i][5], pts[i][6], pts[i][7], pts[i][8],
				  b_rad, b_res, use_cyl);
	       }
	    }
	    // I can draw 3 of the four triangles, but I need
	    // the normals to do the last one properly
	    for (i= 5; i<8-1; i++) {
/*
       cmdTriangle.putdata(pts[0][0+i], pts[1][0+i], pts[1][1+i],
			   tri_norm[0][-1+i], tri_norm[1][-1+i],
			   tri_norm[1][0+i], this);
       cmdTriangle.putdata(pts[0][0+i], pts[1][1+i], pts[0][1+i],
			   tri_norm[0][-1+i], tri_norm[1][0+i],
			   tri_norm[0][0+i], this);
*/
       cmdTriangle.putdata(pts[1][1+i], pts[1][0+i], pts[0][0+i],
			   tri_norm[1][0+i], tri_norm[1][-1+i],
			   tri_norm[0][-1+i], this);
       cmdTriangle.putdata(pts[0][1+i], pts[1][1+i], pts[0][0+i],
			   tri_norm[0][0+i], tri_norm[1][0+i],
			   tri_norm[0][-1+i], this);
            }
	    last_loop = loop;
	 }  // atom 'loop+1' finished
	 
	 // save information for the next loop through, this is needed
	 // so that the cylinders will be smooth and that the normals
	 // will look nice
	 for (i=0; i<=1; i++) {
	    copy(pts[i][0], pts[i][6]);  // remember, because of the spline,
	    copy(pts[i][1], pts[i][7]);  // loop pts[][8] is loop+1 pts[][2]
	    copy(norms[i][0], norms[i][6]);
	    copy(norms[i][1], norms[i][7]);
	 }
      } /// else nothing to draw
   } // gone down the fragment

   free(tmp_ptr);
}

// draw a cubic spline (in this case, using a modified Catmull-Rom basis)
// through the C alpha of the protein and the P of the nucleic acids
void DrawMolItem::draw_tube(float *framepos) {
   if (!mol->is_current()) {
      return;
   }

   float *pos;
   int atomnum, loop;
   float *real_coords = NULL, *coords;
   int *real_idx = NULL, *idx;

   // find out if I'm using lines or cylinders
   float b_rad = atomRep->bond_rad();
   int b_res = atomRep->bond_res();
   int use_cyl = FALSE;
   if (b_res <= 2 || b_rad < 0.01) { // then going to do lines
      cmdMaterials.putdata(FALSE,this);
      cmdLineType.putdata(SOLIDLINE, this);
      cmdLineWidth.putdata(atomRep->line_thickness(), this);
   } else {
      use_cyl = TRUE;
      cmdMaterials.putdata(TRUE, this);
   }
   int num, frag;

   // go through each protein fragment
   for ( frag=0; frag<mol->pfragList.num(); frag++) {
      num = mol->pfragList[frag]->num(); // number of residues
      if (real_coords) {
	 free(real_coords);   // to store a copy of the coords
         free(real_idx);      // and atom indicies
      }
      
      // get memory, and include space for the extra boundary elements
      // okay, this is a memory hog, sue me.
      // This allocates space for all the coordinates as well as a
      // copy of the first coordinate in -1 and -2, and similar
      // duplicates of the final coordinate.  This make the spline
      // code a lot simpler
      //    For the coordinates
      real_coords = (float *) malloc((num+4)*sizeof(float)*3);
      if (!real_coords) {
	 msgErr << "Not enough memory to make a spline" << sendmsg;
	 continue;
      }
      //    For the mapping from index in the frag list ("loop") to
      //    the offset for the give C-alpha in the molecule
      real_idx = (int *) malloc((num+4) * sizeof(int));
      if (!real_idx) {
	 msgErr << "Not enough memory to make a spline." << sendmsg;
	 free(real_coords);
	 real_coords = NULL;
	 continue;
      }
      coords = real_coords + 2*3;
      idx = real_idx + 2;

      // initialize the beginning coordinates to the first carbon-alpha
      int res = (*mol->pfragList[frag])[0]; // pfragList has residue numbers
      atomnum = mol->find_atom_in_residue("CA", res);   // get the atom index
      if (atomnum < 0) {
	 msgErr << "Cannot find the first CA of a protein!!!!!!" << sendmsg;
	 continue;
      }
      pos = framepos + 3*atomnum;     // put the first coords into
      copy(coords-6, pos);            // the initial control pts
      copy(coords-3, pos);
      idx[-2] = -1;
      idx[-1] = -1;

      // go through and set the normal spline constraints
      for (loop=0; loop<num; loop++) {   // for each residue
	 res = (*mol->pfragList[frag])[loop];
	 // find the CA and get and save the coordinates.  If for some
	 // wierd-o reason it doesn't exist, use the previous coordinates
	 atomnum = mol->find_atom_in_residue("CA", res); 
	 if (atomnum >=0) {
	   pos = framepos + 3*atomnum;
	 }
	 copy(coords+loop*3, pos); //  the coordinates
	 idx[loop] = atomnum;      // and the index
      }

      // initialize the final control points to the last element
      copy(coords+3*num, pos);
      copy(coords+3*(num+1), pos);
      idx[num]   = -1;
      idx[num+1] = -1;

      // so I have the C-alpha coordinate.  Now do the spline stuff
      // the loop starts from -1 so it can init things like the color,
      // etc.
      draw_spline_curve(num, coords, idx, use_cyl, b_rad, b_res);
      
   } // going down pfragList

   // now do the same with the nucleic acid backbone
   // however, in this case I use the P as my control points
   // and if there is no P I use the C5'
   for ( frag = 0; frag< mol->nfragList.num(); frag++) {
      num = mol->nfragList[frag]->num();
      if (real_coords){
	 free(real_coords);
	 free(real_idx);
      }
      // get memory
      real_coords = (float *) malloc((num+4) * sizeof(float)*3);
      if (!real_coords) {
	 msgErr << "Not enough memory to make a nucleic acid spline"
		<< sendmsg;
	 continue;
      }
      real_idx = (int *) malloc( (num+4) * sizeof(int));
      if (!real_idx) {
	 msgErr << "Not enough memory to make a nucleic acid spline."
		<< sendmsg;
	 free(real_coords);
	 real_coords = NULL;
	 continue;
      }
      coords = real_coords + 2*3;
      idx = real_idx + 2;
      int res = (*mol->nfragList[frag])[0]; // get first residue number

      // look for a phosphate.  If it isn't there, look for the C5'
      atomnum = mol->find_atom_in_residue("P", res);
      if (atomnum < 0) {  // no phosphate, so use C5'
	 atomnum = mol->find_atom_in_residue("C5'", res);
	 if (atomnum < 0) {
	    atomnum = mol->find_atom_in_residue("C5*", res);
	 }
      }
      if (atomnum < 0) {
	 msgErr << "Found a nucleic acid that doesn't have P or C5'!!!"
		<< "  Can't render it as tube." << sendmsg;
	 continue;
      }
      
      // initialize
      pos = framepos + 3*atomnum;
      copy(coords-6, pos);
      copy(coords-3, pos);
      idx[-2] = idx[-1] = -1;
      
      // now set up the rest of the coordinates
      for (loop=0; loop<num; loop++) {  // for each residue
	 res = (*mol->nfragList[frag])[loop];
	 atomnum = mol->find_atom_in_residue("P", res); // find the phosphate
	 if (atomnum >=0 ) {  // P was found
	    pos = framepos + 3*atomnum;
	 } else {
	    atomnum = mol->find_atom_in_residue("C5'", res); // or the C5'
	    if (atomnum < 0) {
	       atomnum = mol->find_atom_in_residue("C5*", res);
	    }
	    if (atomnum >= 0) {  // C5' found
	       pos = framepos + 3*atomnum;
	    }
	 }
	 // coord found, so save it
	 copy(coords+loop*3, pos);
	 idx[loop] = atomnum;
      }
      // special case if one residue so I get part of a curve
      // (I wonder if this will work; I won't test it)
      if (num == 1) {
	 atomnum = mol->find_atom_in_residue("C3'", res);
	 if (atomnum < 0) atomnum = mol->find_atom_in_residue("C3*", res);
	 if (atomnum >= 0) {
	    pos = framepos + 3*atomnum;
	 }
      }
      
      // and finish up the data by copying the last element a couple time
      copy(coords+3*num, pos);
      copy(coords+3*num+3, pos);
      idx[num] = idx[num+1] = -1;

      // render everything
      draw_spline_curve(num, coords, idx, use_cyl, b_rad, b_res);
   } // going down nfragList

   if (real_coords) {
      free(real_coords);
      free(real_idx);
   }
}

// draw ribbons along the protein backbone
// part of this method taken (with permission) from the 'ribbon.f' in the
// Raster3D package:
//  Merritt, Ethan A. and Murphy, Michael E.P. (1994).
//   "Raster3D Version 2.0, a Program for Photorealistic Molecular Graphics"
//        Acta Cryst. D50, 869-873.

// That method was based on ideas from  Carson & Bugg, J. Molec. Graphics
//   4,121-122 (1986)
void DrawMolItem::draw_ribbons(float *framepos) {
   if (!mol->is_current()) {
      return;
   }
   float *real_coords = NULL, *coords;
   float *real_perps = NULL, *perps;
   int *real_idx = NULL, *idx;
   float *capos, *last_capos;
   float *opos, *last_opos;
   int onum, canum;
   int frag, num;
   // these are the variables used in the Raster3D package
   float a[3], b[3], c[3], d[3], e[3], g[3];  

   // find out if I'm using lines or cylinders
   float b_rad = atomRep->bond_rad()/3.0;
   int b_res = atomRep->bond_res();
   int use_cyl = FALSE;
   if (b_res <= 2 || b_rad < 0.01) { // the representation will be as lines
   } else {
      use_cyl = TRUE;
   }
   cmdMaterials.putdata(TRUE, this);
   float ribbon_width = (float) atomRep->line_thickness();
   if (ribbon_width < 1.0) ribbon_width = 1.0;
   ribbon_width /= 7.0;
   
   // go through each protein and find the CA and O
   // from that, construct the offsets to use (called "perps")
   // then do two splines and connect them
   for (frag = 0; frag<mol->pfragList.num(); frag++) {
      // number of residues in this fragment
      num = mol->pfragList[frag]->num(); 
      if (num < 2) {
// commented out since it got to be a nuisance
//	msgErr << "Cannot draw a ribbon for a protein with only one element"
//		 << sendmsg;
	 continue;
      }
      if (real_coords) {
	 free(real_coords);
	 free(real_idx);
	 free(real_perps);
      }
      real_coords = (float *) malloc((num+4)*sizeof(float)*3);
      real_idx = (int *) malloc((num+4) * sizeof(int));
      real_perps = (float *) malloc((num+4) * sizeof(float)*3);
      coords = real_coords + 2*3;
      idx = real_idx + 2;
      perps = real_perps + 2*3;

      // okay, I've got space for the coordinates, the index translations,
      // and the perpendiculars, now initialize everything
      int res = (*mol->pfragList[frag])[0];
      canum = mol->find_atom_in_residue("CA", res); // get CA 
      onum = mol->find_atom_in_residue("O", res);   //     O
      if (canum < 0) {
	 msgErr << "Cannot find first CA of a protein, so it won't be drawn"
		<< sendmsg;
	 continue;
      }
      capos = framepos + 3*canum;
      opos = framepos + 3*onum;
      last_opos = opos;
      last_capos = capos;
      
      copy(coords-6, capos);
      copy(coords-3, capos);
      idx[-2] = idx[-1] = -1;

      // now go through and set the coordinates and the perps
      e[0] = e[1] = e[2] = 0.0;
      copy(g,e);
      for (int loop=0; loop<num; loop++)
      {
	 res = (*mol->pfragList[frag])[loop];
	 canum = mol->find_atom_in_residue("CA", res);
	 if (canum >= 0) {
	    capos = framepos + 3*canum;
	 }
	 onum = mol->find_atom_in_residue("O", res);
	 if (onum < 0) {
	    onum = mol->find_atom_in_residue("OT1", res);
	 }
	 if (onum < 0) {
	    msgInfo << "Someone's been messing around with the definition "
	    "of a protein!\n" << "Things are going to look messy" << sendmsg;
	    opos = last_opos;
	 } else {
	    opos = framepos + 3*onum;
	 }
	 copy(coords+loop*3, capos);
	 idx[loop] = canum;

         // now I need to figure out where the ribbon goes
	 // A is (pos(CA(res)) - pos(CA(res-1)))
         subtract(a, capos, last_capos);
	 // B is (pos(O(res-1)) - pos(CA(res-1)))
         subtract(b, last_opos, last_capos);
         cross_prod(c, a, b); // C = A x B // defines res-1's peptide plane
         cross_prod(d, c, a); // D = C x A // normal to plane and backbone
         {
            float tempf = dot_prod(d,g);   // if the normal is > 90 degrees
            if (tempf < 0) {               // from the previous one,
               b[0] = -d[0]; b[1] = -d[1]; b[2] = -d[2];  // invert the new one
            } else {
               copy(b, d);
            }
            add(e,g,b);  // average this new vector to the cumulative sum
         }               // of the previous vectors
	 if (dot_prod(e,e) > 0.001) {  // failsafe it here since
	    normalize(e);              //  normalize doesn't
	 }
         if (loop>0) {                 // compute the offset from the
            copy(perps+3*loop, e);     // normal
	    perps[3*loop+0] *= ribbon_width;
	    perps[3*loop+1] *= ribbon_width;
	    perps[3*loop+2] *= ribbon_width;
         }
         copy(g, e);            // make a cumulative sum; cute, IMHO
         last_capos = capos;
	 last_opos = opos;
      }
      // and set the final points to the last element
      copy(coords+3*num, capos);
      copy(coords+3*(num+1), capos);
      idx[num] = idx[num+1] = -1;

      // now set the first and last perps correctly
      copy(perps, perps+3);
      copy(perps-3, perps);
      copy(perps-6, perps);
      copy(perps+3*num, perps+3*num-3);
      copy(perps+3*num+3, perps+3*num);

      draw_spline_ribbon(num, coords, perps, idx, use_cyl,
			 b_rad, b_res);
   } // drew the protein fragment ribbon


///////////// now draw the nucleic acid ribbon
   
   {
      float *ppos, *last_ppos;
      float opos[3], last_opos[3];
      int pnum;
      int o1num, o2num;
      // go through each nucleic acid and find the phospate
      // then find the O1P and O2P. From those construct the perps
      // then do two splines and connect them
      for (frag = 0; frag<mol->nfragList.num(); frag++) {
	 // number of residues in this fragment
	 num = mol->nfragList[frag]->num(); 
	 if (num < 2) {
//	msgErr << "Cannot draw a ribbon for a nucleic acid with only one element"
//		 << sendmsg;
	    continue;
	 }
	 if (real_coords) {
	    free(real_coords);
	    free(real_idx);
	    free(real_perps);
	 }
	 real_coords = (float *) malloc((num+4)*sizeof(float)*3);
	 real_idx = (int *) malloc((num+4) * sizeof(int));
	 real_perps = (float *) malloc((num+4) * sizeof(float)*3);
	 coords = real_coords + 2*3;
	 idx = real_idx + 2;
	 perps = real_perps + 2*3;
	 
	 // okay, I've got space for the coordinates, the index translations,
	 // and the perpendiculars, now initialize everything
	 int res = (*mol->nfragList[frag])[0];
	 pnum = mol->find_atom_in_residue("P", res);  // get phosphate 
	 o1num = mol->find_atom_in_residue("O1P", res); //  and an oxygen
	 o2num = mol->find_atom_in_residue("O2P", res); //  and an oxygen
	 if (pnum < 0) {
	    msgErr << "Cannot find first phosphate of a nucleic acid, so it won't be drawn"
		   << sendmsg;
	    continue;
	 }
	 if (o1num  < 0 || o2num < 0) {
	    msgErr << "Cannot find both O1P and O2P of a nucleic acid, so it won't be drawn"
		   << sendmsg;
	    continue;
	 }
	 ppos = framepos + 3*pnum;
	 add(opos, framepos + 3*o1num, framepos + 3*o2num);  // along the bisector
	 copy(last_opos, opos);
	 last_ppos = ppos;
	 
	 copy(coords-6, ppos);
	 copy(coords-3, ppos);
	 idx[-2] = idx[-1] = -1;
	 
	 // now go through and set the coordinates and the perps
	 e[0] = e[1] = e[2] = 0.0;
	 copy(g,e);
	 for (int loop=0; loop<num; loop++)
	 {
	    res = (*mol->nfragList[frag])[loop];
	    pnum = mol->find_atom_in_residue("P", res);
	    if (pnum >= 0) {
	       ppos = framepos + 3*pnum;
	    }
	    o1num = mol->find_atom_in_residue("O1P", res);
	    o2num = mol->find_atom_in_residue("O2P", res);
	    if (o1num < 0 || o2num < 0) {
	       msgInfo << "Someone's been messing around with the definition "
	       "of a nucleic acid!\n" << "Things are going to look messy" << sendmsg;
	       copy(opos, last_opos);
	    } else {
	       float tmp[3];
	       subtract(tmp, framepos + 3*o1num, ppos);
	       subtract(opos, framepos + 3*o2num, ppos);
	       add(opos, tmp, opos);  // along the bisector
	    }
	    copy(coords+loop*3, ppos);
	    idx[loop] = pnum;
	    
	    // now I need to figure out where the ribbon goes
	    // A is (pos(P(res)) - pos(P(res-1)))
	    subtract(a, ppos, last_ppos);
	    // B is (pos(Obisector(res-1)) - pos(P(res-1)))
//	    subtract(b, last_opos, last_ppos);
//	    cross_prod(c, a, b); // C = A x B // defines res-1's peptide plane
	    copy(c, opos); // already have the normal to the ribbon
	    cross_prod(d, c, a); // D = C x A // normal to plane and backbone
	    {
	       float tempf = dot_prod(d,g);   // if the normal is > 90 degrees
	       if (tempf < 0) {               // from the previous one,
		  b[0] = -d[0]; b[1] = -d[1]; b[2] = -d[2];  // invert the new one
	       } else {
		  copy(b, d);
	       }
            add(e,g,b);  // average this new vector to the cumulative sum
	    }               // of the previous vectors
	    if (dot_prod(e,e) > 0.001) {  // failsafe it here since
	       normalize(e);              //  normalize doesn't
	    }
	    if (loop>0) {                 // compute the offset from the
	       copy(perps+3*loop, e);     // normal
	       perps[3*loop+0] *= ribbon_width;
	       perps[3*loop+1] *= ribbon_width;
	       perps[3*loop+2] *= ribbon_width;
	    }
	    copy(g, e);            // make a cumulative sum; cute, IMHO
	    last_ppos = ppos;
	    copy(last_opos, opos);
	 }
	 // and set the final points to the last element
	 copy(coords+3*num, ppos);
	 copy(coords+3*(num+1), ppos);
	 idx[num] = idx[num+1] = -1;
	 
	 // now set the first and last perps correctly
	 copy(perps, perps+3);
	 copy(perps-3, perps);
	 copy(perps-6, perps);
	 copy(perps+3*num, perps+3*num-3);
	 copy(perps+3*num+3, perps+3*num);
	 
	 draw_spline_ribbon(num, coords, perps, idx, use_cyl,
			    b_rad, b_res);
       // drew the nucleic acid fragment ribbon
      }
   }

///////////// now draw the rings of the base and sugar as planes 
	// ribose 
	// 	O4',C1',C2',C3'C4' or
	// 	O4*,C1*,C2*,C3*C4*
	// purines (CYT,THY,URA) 
	// 	N1,C2,N3,C4,C5,C6
	// pyrimidines (ADE,GUA) 
	//      N1,C2,N3,C4,C5,C6,N7,N8,N9
	// O4',C1' and C4' are define the ribose plane and 
	// C2' and C3' then define the pucker of the ring
   	// sugar -- base bonds
	//  	pyrimidine 	C1' to N9 
	// 	purine 		C1' to N1  
   {
      float *o4ppos, *c1ppos, *c2ppos, *c3ppos, *c4ppos;
      int o4pnum, c1pnum, c2pnum, c3pnum, c4pnum;
      float *n1pos,*c2pos,*n3pos,*c4pos,*c5pos,*c6pos,*n7pos,*c8pos,*n9pos;
      int n1num,c2num,n3num,c4num,c5num,c6num,n7num,c8num,n9num;
      float rescentra[3], rescentrb[3];
      float midptc1pc4p[3];
      
      for (frag = 0; frag<mol->nfragList.num(); frag++) {
	 // number of residues in this fragment
	 num = mol->nfragList[frag]->num(); 
	 if (real_coords) {
	    free(real_coords);
	 }
	 //
	 // 5atoms for the ribose but only 4 triangles
	 // 9atoms max for a base
	 real_coords = (float *) malloc( 14 * num * sizeof(float)*3);
	 coords = real_coords + 2*3;
	 
	 // okay, I've got space for the coordinates now go
	 for (int loop=0; loop<num; loop++){
	 // the furanose
	    int res = (*mol->nfragList[frag])[loop];

	    c1pnum = mol->find_atom_in_residue("C1'", res);
	    if (c1pnum < 0) {
	       c1pnum = mol->find_atom_in_residue("C1*", res);
	    }
	    if ( atomSel->on[c1pnum] ){ // switch drawing by C1' atom

	      o4pnum = mol->find_atom_in_residue("O4'", res);
	      if (o4pnum < 0) {
		 o4pnum = mol->find_atom_in_residue("O4*", res);
	      }
	      c2pnum = mol->find_atom_in_residue("C2'", res); 
	      if (c2pnum < 0) {
		 c2pnum = mol->find_atom_in_residue("C2*", res);
	      }
	      c3pnum = mol->find_atom_in_residue("C3'", res); 
	      if (c3pnum < 0) {
		 c3pnum = mol->find_atom_in_residue("C3*", res);
	      }
	      c4pnum = mol->find_atom_in_residue("C4'", res);
	      if (c4pnum < 0) {
		 c4pnum = mol->find_atom_in_residue("C4*", res);
	      }
	      if (o4pnum < 0 || c2pnum < 0 || c3pnum < 0 || c4pnum < 0) {
		 msgErr << "Someone has redefined the nucleic acid atom names!"
			<< sendmsg;
		 continue;
	      }

	      o4ppos = framepos + 3*o4pnum;
	      c1ppos = framepos + 3*c1pnum;
	      c2ppos = framepos + 3*c2pnum;
	      c3ppos = framepos + 3*c3pnum;
	      c4ppos = framepos + 3*c4pnum;
	 
	      midpoint(midptc1pc4p, c1ppos, c4ppos);
	 	 
	 // now display triangles 
 	    cmdColorIndex.putdata(atomColor->color[c1pnum], this);
          
	    cmdTriangle.putdata(c4ppos,c1ppos,o4ppos,this);
            cmdTriangle.putdata(c3ppos,midptc1pc4p,c4ppos,this);
            cmdTriangle.putdata(c2ppos,midptc1pc4p,c3ppos,this);
            cmdTriangle.putdata(c1ppos,midptc1pc4p,c2ppos,this);
	   }	
	// begin bases
	  rescentra[0]=rescentra[1]=rescentra[2]=0.0;
	  rescentrb[0]=rescentrb[1]=rescentrb[2]=0.0;
	  
	  // check for purine and pyrimidine specific atoms
	  n9num = mol->find_atom_in_residue("N9", res);    	
	  n1num = mol->find_atom_in_residue("N1", res);
	    // if there is a N9, then this is a pyrimidine
	  if(( n9num >= 0) && (atomSel->on[n9num])) {
	
	     	c8num = mol->find_atom_in_residue("C8", res); 
	    	n7num = mol->find_atom_in_residue("N7", res); 
		c6num = mol->find_atom_in_residue("C6", res); 
	    	c5num = mol->find_atom_in_residue("C5", res);
	    	c4num = mol->find_atom_in_residue("C4", res);
	    	n3num = mol->find_atom_in_residue("N3", res);
	    	c2num = mol->find_atom_in_residue("C2", res);
	    	n1num = mol->find_atom_in_residue("N1", res);
		if (c8num < 0 || n7num < 0 || c6num < 0 || c5num < 0 ||
		    c4num < 0 || n3num < 0 || c2num < 0 || n1num < 0) {
 msgWarn << "Tried to draw a nucleic acid residue I thought was a pyrimidine, "
	 << sendmsg;
 msgWarn << "but it doesn't have the right atom names." << sendmsg;
                    continue;
		}

		n9pos = framepos + 3*n9num;    	
	     	add(rescentra,rescentra,n9pos);
		c8pos = framepos + 3*c8num; 
	     	add(rescentra,rescentra,c8pos);
		n7pos = framepos + 3*n7num; 
	     	add(rescentra,rescentra,n7pos);

		c5pos = framepos + 3*c5num;
	     	add(rescentra,rescentra,c5pos);
		add(rescentrb,rescentrb,c5pos);
	    	c4pos = framepos + 3*c4num;
	     	add(rescentra,rescentra,c5pos);
		add(rescentrb,rescentrb,c5pos);
		
		c6pos = framepos + 3*c6num; 
	     	add(rescentrb,rescentrb,c6pos);
	    	n3pos = framepos + 3*n3num;
	     	add(rescentrb,rescentrb,n3pos);
	    	c2pos = framepos + 3*c2num;
	     	add(rescentrb,rescentrb,c2pos);
	    	n1pos = framepos + 3*n1num;
	     	add(rescentrb,rescentrb,n1pos);
		
		rescentrb[0] = rescentrb[0]/6.0;
		rescentrb[1] = rescentrb[1]/6.0;
		rescentrb[2] = rescentrb[2]/6.0;
		
		rescentra[0] = rescentra[0]/5.0;
		rescentra[1] = rescentra[1]/5.0;
		rescentra[2] = rescentra[2]/5.0;

	       // draw bond from ribose to base
	        cmdCylinder.putdata(c1ppos, n9pos, b_rad, b_res, this);
	      // now display triangles
 	 	cmdColorIndex.putdata(atomColor->color[n9num], this);

          	cmdTriangle.putdata(n1pos,rescentrb,c2pos,this);
          	cmdTriangle.putdata(c2pos,rescentrb,n3pos,this);
          	cmdTriangle.putdata(n3pos,rescentrb,c4pos,this);
          	cmdTriangle.putdata(c4pos,rescentrb,c5pos,this);
          	cmdTriangle.putdata(c5pos,rescentrb,c6pos,this);
          	cmdTriangle.putdata(c6pos,rescentrb,n1pos,this);

          	cmdTriangle.putdata(n9pos,rescentra,c8pos,this);
          	cmdTriangle.putdata(c8pos,rescentra,n7pos,this);
          	cmdTriangle.putdata(n7pos,rescentra,c5pos,this);
          	cmdTriangle.putdata(c5pos,rescentra,c4pos,this);
          	cmdTriangle.putdata(c5pos,rescentra,c4pos,this);
          	cmdTriangle.putdata(c4pos,rescentra,n9pos,this);

		}	    
	     else if (( n1num >= 0) && (atomSel->on[n1num])){
	        // residue is purine and turned on

		c6num = mol->find_atom_in_residue("C6", res); 
	    	c5num = mol->find_atom_in_residue("C5", res);
	    	c4num = mol->find_atom_in_residue("C4", res);
	    	n3num = mol->find_atom_in_residue("N3", res);
	    	c2num = mol->find_atom_in_residue("C2", res);
		if (c6num < 0 || c5num < 0 || c4num < 0 || n3num < 0 ||
		    c2num < 0) {
 msgWarn << "Tried to draw a nucleic acid residue I thought was a purine, "
	 << sendmsg;
 msgWarn << "but it doesn't have the right atom names." << sendmsg;
                    continue;
		}

		c6pos = framepos + 3*c6num; 
	     	add(rescentrb,rescentrb,c6pos);
		c5pos = framepos + 3*c5num;
		add(rescentrb,rescentrb,c5pos);
	    	c4pos = framepos + 3*c4num;
		add(rescentrb,rescentrb,c5pos);
	    	n3pos = framepos + 3*n3num;
	     	add(rescentrb,rescentrb,n3pos);
	    	c2pos = framepos + 3*c2num;
	     	add(rescentrb,rescentrb,c2pos);
	    	n1pos = framepos + 3*n1num;
	     	add(rescentrb,rescentrb,n1pos);

		rescentrb[0] = rescentrb[0]/6.0;
		rescentrb[1] = rescentrb[1]/6.0;
		rescentrb[2] = rescentrb[2]/6.0;

 	       // draw bond from ribose to base
	        cmdCylinder.putdata(c1ppos, n1pos, b_rad, b_res, this);
	 	cmdColorIndex.putdata(atomColor->color[n1num], this);

          	cmdTriangle.putdata(n1pos,rescentrb,c2pos,this);
          	cmdTriangle.putdata(c2pos,rescentrb,n3pos,this);
          	cmdTriangle.putdata(n3pos,rescentrb,c4pos,this);
          	cmdTriangle.putdata(c4pos,rescentrb,c5pos,this);
          	cmdTriangle.putdata(c5pos,rescentrb,c6pos,this);
          	cmdTriangle.putdata(c6pos,rescentrb,n1pos,this);
		}
	
	}
       
     }
  }
     if (real_coords) {
      free(real_coords);
      free(real_idx);
      free(real_perps);
   }
}


#ifdef VMDALPHA_SHAPE

void DrawMolItem::draw_alphaShape(float *framepos) {
   int count = 0;
   float radius = atomRep->sphere_rad();
   // The value for get_faces was determined empirically so that
   // 1 was a reasonable value and 2 is at or near the max.
   // Without the special mods, the rough range was from 0 to 100000.
   {
      float r = radius / 2.0;
      if (r > 1.0) r = 1.0;
      if (r < 0) r = 0;
      // bring everything closer to 1
      r = sqrt(sqrt(r));
      r = exp(r * 11.513 /* = ln(100001) */ ) - 1;
//      msgInfo << "get_faces radius = " << r << sendmsg;	 
      radius = r;
   }
   int *map = new int[mol->nAtoms];  // mapping from alpha shape index
                                       // to molecular index
   // the ! new option means if only the radius was changed
   // then don't recalculate the alpha shape
//   msgInfo << "alpha == " << 0+(atomRep->alpha==NULL)
//	   << "  radius " << radius << sendmsg;
   if (atomRep -> alpha == NULL ||
       needRegenerate & FORCE_REGEN ||
       needRegenerate & SEL_REGEN  ||
       (needRegenerate & REP_REGEN && atomRep->option_changed)) {
      float *x = new float[mol->nAtoms];
      float *y = new float[mol->nAtoms];
      float *z = new float[mol->nAtoms];
      float *r = new float[mol->nAtoms];
                                    
      // check all atoms if they are displayed, and if so, get
      // the data for it
      for (int i=0; i<mol->nAtoms; i++) {
	 if (atomSel->on[i]) {
	    map[count] = i;
	    x[count] = framepos[3*i + 0];
	    y[count] = framepos[3*i + 1];
	    z[count] = framepos[3*i + 2];
	    r[count] = mol->atom(i)->radius();
	    count ++;
	 }
      }
      atomRep->set_alpha(new AlphaShape(radius, count, x, y, z, r));
      if (!atomRep->alpha->okay()) {
	 atomRep->delete_alpha();
      }
      delete [] x;
      delete [] y;
      delete [] z;
      delete [] r;
   } else { // just get the mapping
      for (int i=0; i<mol->nAtoms; i++) {
	 if (atomSel->on[i]) {
	    map[count++] = i;
	 }
      }
      // and recalculate based on previous data
      atomRep->alpha->get_faces(radius);
      if (!atomRep->alpha->okay()) {
	 atomRep->delete_alpha();
      }
   }
   atomRep -> option_changed = FALSE;

   // if there were any selected atoms for an alpha
   if (atomRep->alpha && count > 0) {  
      cmdMaterials.putdata(TRUE,this);
      // find out which verticies have been selected so I can do
      // labels, if needed
      int *vrtx = new int[mol->nAtoms];
      for (int i=0; i<mol->nAtoms; i++) {
	 vrtx[i]=0;
      }
      // play funny with the triangle - break it up into three smaller
      // quadrilaterals so that each is colored by the closest corner
      float middle[3];
      float sides[3*3];
      float *ptr1, *ptr2, *ptr3;
      int a;
      for (i=0; i<atomRep->alpha->num_triangles; i++) {
	 vrtx[a = map[atomRep->alpha->triangles[i][0]-1]] = 1;
	 ptr1 = framepos + 3*a;
	 vrtx[a = map[atomRep->alpha->triangles[i][1]-1]] = 1;
	 ptr2 = framepos + 3*a;
	 vrtx[a = map[atomRep->alpha->triangles[i][2]-1]] = 1;
	 ptr3 =  framepos + 3*a;

	 add(middle, ptr1, ptr2);
	 copy(sides, middle);
	 add(middle, middle, ptr3);
	 middle[0]/=3; middle[1]/=3; middle[2]/=3;
	 add(sides+3, ptr2, ptr3);
	 add(sides+6, ptr3, ptr1);
	 for(int j=0; j<9; j++) {
	    sides[j]/=2;
	 }
	 
	 cmdColorIndex.putdata(atomColor->
		     color[map[atomRep->alpha->triangles[i][0]-1]], this);
	 cmdTriangle.putdata(ptr1, sides, middle, this);
	 cmdTriangle.putdata(sides+6, ptr1, middle, this);

	 cmdColorIndex.putdata(atomColor->
		     color[map[atomRep->alpha->triangles[i][1]-1]], this);
	 cmdTriangle.putdata(ptr2, sides+3, middle, this);
	 cmdTriangle.putdata(sides, ptr2, middle, this);

	 cmdColorIndex.putdata(atomColor->
		     color[map[atomRep->alpha->triangles[i][2]-1]], this);
	 cmdTriangle.putdata(ptr3, sides+6, middle, this);
	 cmdTriangle.putdata(sides+3, ptr3, middle, this);
      }
      // also draw the edges
      float *f1, *f2;
      for (i=0; i<atomRep->alpha->num_edges; i++) {
	 cmdColorIndex.putdata(atomColor->
			       color[map[atomRep->alpha->edges[i][0]-1]],
			       this);
	 add(middle, f1 = (framepos + 3*map[atomRep->alpha->edges[i][0]-1]),
	             f2 = (framepos + 3*map[atomRep->alpha->edges[i][1]-1]));
	 middle[0]/=2; middle[1]/=2; middle[2]/=2;
	 cmdLine.putdata(f1, middle, this);
	 cmdLine.putdata(middle, f2, this);
      }
      // finally, display any selected atoms
      for (i=0; i<mol->nAtoms; i++) {
	 if (vrtx[i]) {
	    pickPointIndex.putdata(3*i, i, this);
	 }
      }
      delete vrtx;
   }
   delete [] map;
   msgInfo << "Alpha shape drawn." << sendmsg;
}
#endif /* VMDALPHA_SHAPE */

