/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: FileRenderer.C,v $
 *	$Author: khamer $	$Locker:  $		$State: Exp $
 *	$Revision: 1.12 $	$Date: 1995/05/30 13:53:33 $
 *
 ***************************************************************************
 * DESCRIPTION:
 * 
 * The FileRenderer class implements the data and functions needed to 
 * render a scene to a file in some format (postscript, raster3d, etc.)
 *
 ***************************************************************************/

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "FileRenderer.h"
#include "Stack.h"
#include "Matrix4.h"
#include "DispCmds.h"
#include "Inform.h"
#include "utilities.h"

///////////////////////// constructor and destructor

// constructor
FileRenderer::FileRenderer(char *nm) : DisplayDevice(nm) {
  need_aaon = need_cueon = FALSE;
  dataBlock = NULL;
  
  currentColor[0] = 0.0;
  currentColor[1] = 1.0;
  currentColor[2] = 0.0;
  
  sprintf(renderCommand, "");
  sprintf(fname,"plot");
  
  MSGDEBUG(1,"Creating FileRenderer ..." << sendmsg);
}

// destructor
FileRenderer::~FileRenderer(void) { }

///////////////////////// protected nonvirtual routines

// set the line width
void FileRenderer::set_line_width(int w) {
    
    if(w>0) {
        lineWidth = w;
    } else {
        lineWidth = 1;
    }
}

// set the line style
void FileRenderer::set_line_style(int s) {
    if(s == ::SOLIDLINE || s == DASHEDLINE) {
        lineStyle = s;
    } else {
        lineStyle = ::SOLIDLINE;
        msgErr << "FileRenderer: Unknown line style " << lineStyle << sendmsg;
    }
}

// set the color
void FileRenderer::set_color(float *rgb) {

  currentColor[0] = rgb[0];
  currentColor[1] = rgb[1];
  currentColor[2] = rgb[2];

}

// define a material
int FileRenderer::do_define_material(int n, float *data) {

  // For the moment, just return true.  Look at GLRenderer.[Ch]
  // for more info.
  n = n;
  data = data;

  return TRUE;
}


// define a light
void FileRenderer::define_light(int i, float *a, float *b) {

  // not implemented
  i = i;
  a = a;
  b = b;

}

// turn a light on or off
void FileRenderer::light_onoff(int i, int j) {

  // not implemented
  i = i;
  j = j;

}

// scale the radius a by the global scaling factor, return as b.
float FileRenderer::scale_radius(float r) {
  // of course, VMD does not have a direction-independent scaling
  // factor, so I'll fake it using an average of the scaling
  // factors in each direction.
  
  float scaleFactor;
  
  scaleFactor = (sqrtf( 
  (((transMat.top()).mat)[0][0])*(((transMat.top()).mat)[0][0]) +
  (((transMat.top()).mat)[1][0])*(((transMat.top()).mat)[1][0]) +
  (((transMat.top()).mat)[2][0])*(((transMat.top()).mat)[2][0]) ) +
  sqrtf( (((transMat.top()).mat)[0][1])*(((transMat.top()).mat)[0][1]) +
  (((transMat.top()).mat)[1][1])*(((transMat.top()).mat)[1][1]) +
  (((transMat.top()).mat)[2][1])*(((transMat.top()).mat)[2][1]) ) +
  sqrtf( (((transMat.top()).mat)[0][2])*(((transMat.top()).mat)[0][2]) +
  (((transMat.top()).mat)[1][2])*(((transMat.top()).mat)[1][2]) +
  (((transMat.top()).mat)[2][2])*(((transMat.top()).mat)[2][2]) ) ) / 3.0;
  
  if(r < 0.0) {
    msgErr << "FileRenderer: Error, Negative radius" << sendmsg;
    r = -r;
  } 
  
  r = r * scaleFactor;
  
  return r;
}


/////////////////////// protected virtual routines

   void FileRenderer::point(float * ){}
   void FileRenderer::sphere(float * ){}
   void FileRenderer::line(float *, float * ) {}
   void FileRenderer::cylinder(float *, float *, float ) {}
   void FileRenderer::cone(float *, float *, float ) {}
   void FileRenderer::triangle(float *, float *, float *, float *, float *, float
   *) {}
   void FileRenderer::square(float *, float *, float *, float *, float *) {}
   void FileRenderer::text(char *){}
   void FileRenderer::text_position(float , float , float ) {}
   void FileRenderer::activate_materials(void) {}
   void FileRenderer::deactivate_materials(void) {}

/////////////////////// public virtual routines

// set the filename
void FileRenderer::set_filename(char * filename) {
  strcpy(fname,filename);
}

const char *FileRenderer::get_filename(void) {
  return fname;
}

// return the default render command line
const char *FileRenderer::default_render(void) {
  return renderCommand;
}

// render the command list to the file in order.
void FileRenderer::render(void *cmdlist) {
  char *cmdptr = (char *)cmdlist;
  int tok, cmdsize;
  
  MSGDEBUG(3, "FileRenderer: rendering command list." << sendmsg);
  
  if (!cmdlist) return;
  
  // scan through the list, getting each command and executing it,
  // until the end of the commands is found
  dataBlock = NULL;
  while((tok = ((int *)cmdptr)[0]) != DLASTCOMMAND) {
    if(tok == DLINKLIST) {
       cmdptr += sizeof(int);       // step forward one (no length here)
       cmdptr = *((char **) cmdptr);// and go to the next link
       tok = ((int *)cmdptr)[0];    // this is guarenteed to be neither
                                    // a DLINKLIST nor DLASTCOMMAND
    }
    
    cmdsize = ((int *)cmdptr)[1];
    cmdptr += 2*sizeof(int);
    
    MSGDEBUG(4, "FileRenderer: doing command " << tok);
    MSGDEBUG(4, " of size " << cmdsize << sendmsg );
    
    if(tok == DPOINT) {
    // plot a point (a sphere of size defaultRadius)
      point((float *)cmdptr);   
 
    } else if(tok == DPOINT_I) {
    // plot a point using indices into dataBlock
      point(dataBlock + ((int *)cmdptr)[0]);
 
    } else if(tok == DSPHERE) {
    // plot a sphere
      sphere((float *)cmdptr);
 
    } else if(tok == DSPHERE_I) {
    // plot a sphere using indices into dataBlock
      int n1 = (int)(((float *)cmdptr)[0]);
      float spheredata[4];
      spheredata[0] = dataBlock[n1++];
      spheredata[1] = dataBlock[n1++];
      spheredata[2] = dataBlock[n1];
      spheredata[3] = ((float *)cmdptr)[1];
      sphere(spheredata);

    } else if(tok == DLINE) {
    // plot a line
      line((float *)cmdptr, ((float *)cmdptr) + 3);
    
    } else if(tok == DLINE_I) {
    // three guesses and the first two don't count
      line(dataBlock + ((int *)cmdptr)[0], dataBlock + ((int *)cmdptr)[1]);
    
    } else if(tok == DCYLINDER) {
    // plot a cylinder
      cylinder((float *)cmdptr, ((float *)cmdptr) + 3, ((float *)cmdptr)[6]);
    
    } else if(tok == DCYLINDER_I) {
    // more cylinder
      cylinder(dataBlock + (int)(((float *)cmdptr)[0]), dataBlock +
      (int)(((float *)cmdptr)[1]), ((float *)cmdptr)[2] );
    
    } else if(tok == DCONE) {
    // plot a cone  
      cone((float *)cmdptr, ((float *)cmdptr) + 3, ((float *)cmdptr)[6]);
    
    } else if(tok == DCONE_I) {
    // plot a cone, again.
      cone(dataBlock + (int)(((float *)cmdptr)[0]), dataBlock +
      (int)(((float *)cmdptr)[1]), ((float *)cmdptr)[2] );
    
    } else if(tok == DTRIANGLE) {
    // plot a triangle
       triangle((float *)cmdptr, ((float *)cmdptr) + 3, ((float *)cmdptr) +  6,
       ((float *)cmdptr) + 9, ((float *)cmdptr) + 12, ((float *)cmdptr) + 15);
    
    } else if(tok == DTRIANGLE_I) {
    // plot a triangle
       triangle( dataBlock + ((int *) cmdptr)[1],
		 dataBlock + ((int *) cmdptr)[2],
		 dataBlock + ((int *) cmdptr)[3],
		 dataBlock + ((int *) cmdptr)[0],
		 dataBlock + ((int *) cmdptr)[0],
		 dataBlock + ((int *) cmdptr)[0] 
		 );
    
    } else if(tok == DSQUARE) {
    // plot a square
      square((float *)cmdptr, ((float *)cmdptr) + 3, ((float *)cmdptr) + 6, 
      ((float *)cmdptr) + 9, ((float *)cmdptr) + 12);
    
    } else if(tok == DSQUARE_I) {
    // plot a square
      square(dataBlock + ((int *)cmdptr)[0], dataBlock + ((int*)cmdptr)[1], 
      dataBlock + ((int *)cmdptr)[2], dataBlock + ((int *)cmdptr)[3], 
      dataBlock + ((int *)cmdptr)[4]);
    
    } else if(tok == DLINEWIDTH) {
    // set the width of lines
      set_line_width(((int *)cmdptr)[0]);
    
    } else if(tok == DLINESTYLE) {
    // set line type
      set_line_style(((int *)cmdptr)[0]);
        
    } else if(tok == DCOLORINDEX) {
        int n = ((int *)cmdptr)[0];
        
        if(!matDefined[n]) {
          msgErr << "FileRenderer: Error, color " << n << " not defined."
          <<sendmsg;
        } else {
          set_color(&(matData[n][COLOR_INDEX]));
        }
          
    } else if(tok == DCOLORRGB) {
    // set the color
      set_color((float *)cmdptr);

    } else if(tok == DCOLORDEF) {
    // define a new material
      int n = (int)(((float *)cmdptr)[0]);
      define_material(n, ((float *)cmdptr) + 1);
    
    } else if(tok == DPUSH) {
    // push the current xform matrix
    transMat.push();
    viewChanged = 1;
    
    } else if(tok == DPOP) {
    // pop the current xform matrix
    transMat.pop();
    viewChanged = 1;
    
    } else if(tok == DLOAD) {
    Matrix4 tmpmat((float *)cmdptr);
    (transMat.top()).loadmatrix(tmpmat);
    viewChanged = 1;

    } else if(tok == DMULT) {
    // multiply the given xform matrix
    Matrix4 tmpmat((float *)cmdptr);
    (transMat.top()).multmatrix(tmpmat);
    viewChanged = 1;
       
    } else if(tok == DTRANS) {
    // do a translation
    (transMat.top()).translate(((float *)cmdptr)[0], ((float *)cmdptr)[1],
      		((float *)cmdptr)[2]);
    viewChanged = 1;
    
    } else if(tok == DROT) {
    // do a rotation about a given axis
    (transMat.top()).rot(((float *)cmdptr)[0],
      		'x' + (int)(((float *)cmdptr)[1]));
    viewChanged = 1;

    } else if(tok == DSCALE) {
    // do a scale
    (transMat.top()).scale(((float *)cmdptr)[0], ((float *)cmdptr)[1],
      		((float *)cmdptr)[2]);
    viewChanged = 1;
    
    } else if(tok == DMOREDATA || tok == DNEWDATA) {
    // set the current drawing data block
#ifdef VMDCAVE
    dataBlock = (float *)cmdptr;             // point to the current position in list
#else
    dataBlock = ((float **)cmdptr)[0];       // point to given memory loc
#endif
    
    } else if(tok == DTEXT) {
    // display text at the current position
    text(cmdptr);
    
    } else if(tok == DTEXTPOS) {
    // set the current text position
    text_position(((float *)cmdptr)[0], ((float *)cmdptr)[1], 
    ((float *)cmdptr)[2]);
    
    } else if(tok == DTEXTPOS_I) {
    // same as above with indices into the datablock
    int n1 = ((int *)cmdptr)[0];
    text_position((float)dataBlock[n1], (float)dataBlock[n1+1], 
    (float)dataBlock[n1+2]);
    
    } else if(tok == DLIGHTDEF) {
    // define a light source
    int n = (int)(((float *)cmdptr)[0]);
    define_light(n, ((float *)cmdptr) + 1, ((float *)cmdptr) + 4);
    
    } else if(tok == DLIGHTONOFF) {
    // turn a light on or off
    light_onoff(((int *)cmdptr)[0], ((int *)cmdptr)[1]);
    
    } else if(tok == DSPHERERES) {
    
    // Do nothing
    
    } else if(tok == DSPHERETYPE) {
    
    // Do nothing
    
    } else if(tok == DMATERIALS) {
    // enable or disable materials characteristics
    if(((int *)cmdptr)[0])
      activate_materials();
    else
      deactivate_materials();
    
    } else if(tok == DCLEAR) {

    clear();
    
    } else if(tok == DPICKPOINT) {
    
    // do nothing, this is not part of a file to be rendered.
        
    } else if(tok == DPICKPOINT_I) {
    
    // do nothing  
      
    } else if(tok == DPICKLINE) {
    
    // do nothing  
      
    } else if(tok == DPICKLINE_I) {
    
    // do nothing  
      
    } else if(tok == DPICKBOX) {
    
    // do nothing  
      
    } else if(tok == DPICKBOX_I) {
    
    // do nothing  
      
    } else {

      // command not found
      msgErr << "FileRenderer: Unknown drawing token " << tok
             << " encountered .. skipping this command." << sendmsg;
             
    }
    
    // update position in array
    cmdptr += cmdsize;
    
  }

}

