/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: Global.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.51 $	$Date: 1995/05/27 06:49:27 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Global data structures, configuration settings, and macros for the 
 * vmd library.
 *
 * Where most 'optional components' are processed ... options which are not
 * included are ifdef'd out.
 ***************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

// project include files
#include "Global.h"
#include "Inform.h"
#include "VMDTitle.h"
#include "DisplayDevice.h"
#include "NormalScene.h"
#include "ColorList.h"
#include "LightList.h"
#include "CommandQueue.h"
#include "UIList.h"
#include "UIText.h"
#include "UIVR.h"
#include "Stage.h"
#include "Axes.h"
#include "TrackerList.h"
#include "MoleculeList.h"
#include "Mouse.h"
#include "ConfigList.h"
#include "GeometryList.h"
#include "FileRenderList.h"
#include "startup.h"

// CAVE-specific files
#ifdef VMDCAVE
#include "cave.h"
#include "CaveDisplayDevice.h"
#include "CaveRoutines.h"
#include "CaveScene.h"
#endif

// GL-specific files
#ifdef VMDGL
#include "GLDisplayDevice.h"
#endif

// Starbase-specific files                                     
#ifdef VMDSTARBASE
#include "SBDisplayDevice.h"
#endif                    
                            
// use FORMS GUI?  If so, include GL and FORMS-specific objects
#ifdef VMDFORMS
#include "AnimateFormsObj.h"
#include "ColorFormsObj.h"
#include "DisplayFormsObj.h"
#include "EditFormsObj.h"
#include "FilesFormsObj.h"
#include "GeometryFormsObj.h"
#include "GraphicsFormsObj.h"
#include "MainFormsObj.h"
#include "MolFormsObj.h"
#include "RenderFormsObj.h"
#include "TrackerFormsObj.h"

#ifdef VMDREMOTE
#include "RemoteFormsObj.h"
#include "SimFormsObj.h"
#endif

#endif

//
// main global variables
//

char *myName=NULL;		// name of this program
char *myPath=NULL;		// directory location of this program
char *myTempDir=NULL;		// directory for temp file storage
char *dataPath=NULL;		// where VMD data files are found
char *vmdBabelBin=NULL;         // location of the Babel executable
char *vmdHTMLViewer=NULL;       // name of HTML viewer to use
DisplayDevice *display=NULL;	// where the images are rendered
Scene *scene=NULL;		// the list of all Displayable objects to draw
LightList *lights=NULL;		// the list of all Lights to be drawn
ColorList *colors=NULL;			// the list of all color objects
Axes *axes=NULL;			// set of axes shown on display
Stage *stage=NULL;			// stage displayed below objects
CommandQueue *commandQueue=NULL;	// the command processor
MoleculeList *moleculeList=NULL;	// list of all molecules
GeometryList *geometryList=NULL;	// list of all geometry monitors
SymbolTable atomSelParser;              // functions for atom selection
VMDTitle *vmdTitle = NULL;		// the spinning title
ConfigList *configList = NULL;		// the configuration list
FileRenderList *fileRenderList = NULL;  // the ways to render scenes to file
UIList *uiList=NULL;			// collection of all UI objects
UIVR *uiVR=NULL;			// 3D user interface
UIText *uiText = NULL;			// text interface
Mouse *mouse = NULL;			// mouse interface

#ifdef VMDTK
#include "UITk.h"       
UITk *uiTk = NULL;                      // Tk interface   
#endif

#ifdef VMDREMOTE
#include "RemoteList.h"
Remote *remote = NULL;		// current remote connection setup
RemoteList *remoteList = NULL;	// list of running simulations
#endif

#ifdef VMDEXTERNAL
#include "UIExternal.h"
UIExternal *uiExternal = NULL;          // for "external" interfaces
#endif

/*****************************************************************/

//
// global routines
//

// print out program title
void VMDtitle(void) {
  msgInfo << VERSION_MSG << "\n" << VMD_AUTHORS;
  msgInfo << "\n-------------------------------------------------\n";
  msgInfo << sendmsg;
}


// given a string, return a new one with the temp dir name prepended.
// This will use an internal static buffer, so it is NOT reentrant.
char *VMDtempFile(const char *s) {
  static char tmpfilebuf[1024];
  
  // copy in temp string
  strcpy(tmpfilebuf, myTempDir);
  strcat(tmpfilebuf, "/");
  strncat(tmpfilebuf, s, 1022 - strlen(myTempDir));
  tmpfilebuf[1023] = '\0';

  // return converted string
  return tmpfilebuf;
}


// initialization routine for the library globals
void VMDinit(int argc, char *argv[]) {

  // initialize Inform objects ... must be done FIRST
  msgInfo.need_newline(TRUE);
  msgWarn.need_newline(TRUE);
  msgErr.need_newline(TRUE);
  msgDebug.need_newline(TRUE);
#ifdef VMDDEBUG
  msgDebug.on(TRUE);
  msgDebug.output_level(2);
#else
  msgDebug.on(FALSE);
  msgDebug.output_level(1);
#endif

  MSGDEBUG(1, "Beginning main initialization; " << argc - 1);
  MSGDEBUG(1, " command-line parameter(s)." << sendmsg);

  // initialize name and path variables
  breakup_filename(argv[0], &myPath, &myName);
  
  //
  // process environment variables
  //
  VMDgetEnvironment();

  //
  // parse the command-line options
  //
  VMDparseCommandLine(argc, argv);

  // reads the definitions in the given init file at program start.
  VMDreadInit();

  //
  // print out title of program
  //
  if(showTitle == TITLE_ON)
    VMDtitle();

  //
  // create graphics context and global graphics objects
  //

  // initialize the display
  display = NULL;
  int *dloc = (displayLoc[0] > 0 ? displayLoc : NULL);
  int *dsiz = (displaySize[0] > 0 ? displaySize : NULL);
  if(dloc || dsiz);		// keep compiler happy if no GUI or graphics

  // check for a CAVE display
  if (which_display == DISPLAY_CAVE) {
#ifdef VMDCAVE
    CAVEConfigure(&argc, argv, NULL);
    grab_CAVE_memory(80); // 80 megs of memory, at least initially
    display = new CaveDisplayDevice;
#else
    msgErr << "Not compiled with the CAVE options set." << sendmsg;
    which_display = DISPLAY_WIN;
#endif
  }
  
  // check for a standard monitor display
  if(which_display == DISPLAY_WIN) {
#ifdef VMDGL
    display = new GLDisplayDevice(dsiz, dloc);
#elif defined(VMDSTARBASE)
    display = new SBDisplayDevice(dsiz, dloc);
#else
    // here would go command to create non-GL window display, i.e. X
    // since not available yet, switch to using text
    which_display = DISPLAY_TEXT;
#endif
  }
  
  // still no display?  if not, create 'empty' display
  if(which_display == DISPLAY_TEXT || !display) {
    // neither the CAVE nor regular window is to be used ...
    // create a 'default' display, which really does nothing but can be used
    // to test the program
    which_display = DISPLAY_TEXT;
    display = new DisplayDevice("Default Display");
  }

  // set display screen config, if possible
  display->screen_height(displayHeight);
  display->distance_to_screen(displayDist);

  //
  // create the scene
  //

  scene = NULL;

#ifdef VMDCAVE
  if (which_display == DISPLAY_CAVE) {
    scene = new CaveScene;
    msgInfo << "Cave scene created." << sendmsg;
    CAVEInit();  // split into the various drawing processes
    msgInfo << "CAVE Initialized" << sendmsg;
    CAVEDisplay(cave_renderer, 0); // set the fctn ptr for the renders
    msgInfo << "There are " << CAVENumRenderProcesses() <<
      " children." <<  sendmsg;
  }
#endif

  // make sure there is a scene object.  If none yet, create default one.
  if(!scene)
    scene = new NormalScene;

  //
  // create other global objects for the program
  //

  // create color and light lists; turn on lights 0 and 1
  colors = new ColorList(scene);
  lights = new LightList(scene);
  lights->set(0);
  lights->current()->on();
  lights->set(1);
  lights->current()->on();

  // create other useful graphics objects, and tell them what colors to use
  axes = new Axes(scene);
  stage = new Stage(scene);
  stage->off();			// by default, stage is not displayed

  vmdTitle = new VMDTitle(scene);
  if(showTitle == TITLE_OFF)
    vmdTitle->off();

  // initialize the parser with the routines that can access
  // the BaseMolecule data structures.  This is located in
  // AtomSel.C
  atomSelParser_init();
  
  // create the list of molecules (initially empty)
  moleculeList = new MoleculeList(scene);

  // create the list of geometry monitors (initially empty)
  geometryList = new GeometryList(scene);

  // make the classes which can render scenes to different file formats
  fileRenderList = new FileRenderList;

  // tell all these new objects to use the color list we created
  display->use_colors(colors);
  axes->use_colors(colors);
  stage->use_colors(colors);
  geometryList->use_colors(colors);
  moleculeList->use_colors(colors);		// MUST be called right after
						// this object is created!

#ifdef VMDREMOTE
  // create the list of running simulations (initially empty)
  remoteList = new RemoteList;
#endif

  //
  // initialize UI ... creates UIList and CommandQueue, trackers and tools
  //
  VMDinitUI();
  
  //
  // flush command queue, and make system ready to enter event loop
  //
  VMDupdate(FALSE);
  
  // set UIText to read in startup file, if possible, and execute any
  // initial commands.
  if (uiText)
    VMDreadStartup();
}


// initialize the user interface
void VMDinitUI(void) {

  MSGDEBUG(1, "Creating UI objects ..." << sendmsg);

  // create the user interface list ... all UI's register with this object
  uiList = new UIList;
  
  // create command queue
  commandQueue = new CommandQueue(uiList);

  // text user interface
#ifndef VMDTK
  uiText = new UIText(uiList, commandQueue);  
#endif
  
  // mouse user interface
  mouse = new Mouse(uiList, commandQueue, display);

  // create the 3D (virtual reality-like) user interface
  uiVR = new UIVR(uiList, commandQueue);

  // create graphical user interfaces, if necessary
#ifdef VMDGUI
  if(which_display == DISPLAY_WIN) {

    // if we are using the FORMS library ...
#ifdef VMDFORMS
    new AnimateFormsObj(uiList, commandQueue, FALSE, TRUE);
    new ColorFormsObj(uiList, commandQueue, FALSE, TRUE);
    new DisplayFormsObj(uiList, commandQueue, FALSE, TRUE);
    new EditFormsObj(uiList, commandQueue, FALSE, TRUE);
    new FilesFormsObj(uiList, commandQueue, FALSE, TRUE);
    new GeometryFormsObj(uiList, commandQueue, FALSE, TRUE);
    new GraphicsFormsObj(uiList, commandQueue, FALSE, TRUE);
    new MainFormsObj(uiList, commandQueue, FALSE, TRUE);
    new MolFormsObj(uiList, commandQueue, FALSE, TRUE);
    new RenderFormsObj(uiList, commandQueue, FALSE, TRUE);
    new TrackerFormsObj(uiList, commandQueue, FALSE, TRUE);
#ifdef VMDREMOTE
    new RemoteFormsObj(uiList, commandQueue, FALSE, TRUE);
    new SimFormsObj(uiList, commandQueue, FALSE, TRUE);
#endif
#endif /*VMDFORMS*/

    // if using Tk for the GUI
#ifdef VMDTK
    uiTk = new UITk(uiList, commandQueue);
#ifdef VMDSTARBASE
    // Bind the Tk interpreter to the SBDisplayDevice, creating
    //        the main display window.
    ((SBDisplayDevice *)display)->bind_Tk(uiTk);
#endif
#endif /*VMDTK*/
  }
#endif

  // add in the External interface, only if VMDEXTERNAL is defined
#ifdef VMDEXTERNAL
  uiExternal = new UIExternal(uiList, commandQueue);
#endif

  MSGDEBUG(1, "Initializing UI objects ..." << sendmsg);

  // initialize all the interfaces
  uiList->init_UI();
}


// redraw the screen and update all things that need updatin'
int VMDupdate(int check_for_events) {
			
  MSGDEBUG(3, "Beginning main event loop processing ..." << sendmsg);

  if(uiList) {
    if(check_for_events)
      uiList->check_event_UI();
    commandQueue->execute_all(uiList);
    uiList->update_UI();
  } else {
    msgErr << "Help!  No UIList!!" << sendmsg;
    return FALSE;
  }

  if(scene && display) {
    scene->prepare(display);
    if (which_display != DISPLAY_CAVE) {  // in the CAVE, renders are done
      scene->draw(display);                  // by spawned processes
    }
  } else {
    msgErr << "Help!  No Scene OR DisplayDevice!!" << sendmsg;
    return FALSE;
  }
  
  // turn off the spinning vmd when molecules are loaded
  if (! vmdTitle->turned_off_once() && moleculeList->num() > 0)
    vmdTitle->off();

  return TRUE;
}


// given a string, start up a HTML viewer with the string as an argument.
// Return success.
int VMDreadHTML(char *cmd) {
  static pid_t childProc = (-1);

  if(!vmdHTMLViewer || strlen(vmdHTMLViewer) < 1) {
    msgErr << "No HTML viewer has been specified.  Set the environment ";
    msgErr << "variable VMDHTMLVIEWER to the name of a viewer installed ";
    msgErr << "on your system." << sendmsg;
    return FALSE;
  }

  msgInfo << "Loading help file ..." << sendmsg;

  // if a child process has already been started, kill it first ...
  if(childProc > 0)
    kill(childProc, SIGHUP);

  // fork off a new process now ...
  childProc = fork();

  // and deal with the three possible cases.
  if(childProc > 0) {
    // case 1: this is the parent ... nothing to do, return success.

  } else if(childProc == 0) {
    // case 2: this is the child ... exec the viewer if possible
    int retval = execlp(vmdHTMLViewer, vmdHTMLViewer, cmd, (char *)NULL);

    // if we're here, the exec failed ...
    msgErr << "Unable to start HTML viewer '" << cmd << "': cannot load.";
    msgErr << sendmsg;
    exit(1);

  } else {
    // case 3: the fork failed
    msgErr << "Unable to start HTML viewer '" << cmd << "': cannot fork.";
    msgErr << sendmsg;
    return FALSE;
  }

  // if here, we were successful.
  return TRUE;
}


// exit the program normally; first delete all the necessary objects
void VMDexit(char *exitmsg, int exitcode) {

  msgInfo << VERSION_MSG << "\n";
  if(exitmsg)
    msgInfo << exitmsg << sendmsg;
  else
    msgInfo << "Exiting ..." << sendmsg;

  // delete all objects we created during initialization

#ifdef VMDREMOTE
  // create the list of running simulations (initially empty)
  if(remoteList) delete remoteList;
  remoteList = NULL;
#endif

#ifdef VMDEXTERNAL
  if(uiExternal) delete uiExternal;
#endif

#ifdef VMDTK
  if(uiTk) delete uiTk;
#endif

  if(fileRenderList) delete fileRenderList;
  if(trackerList) delete trackerList;
  if(mouse) delete mouse;
  if(uiText) delete uiText;
  if(uiList) delete uiList;
  if(uiVR) delete uiVR;
  if(geometryList) delete geometryList;
  if(moleculeList) delete moleculeList;
  if(commandQueue) delete commandQueue;
  if(vmdTitle) delete vmdTitle;
  if(stage) delete stage;
  if(axes) delete axes;
  if(lights) delete lights;
  if(colors) delete colors;
  if(scene) delete scene;
  if(display) delete display;
#ifdef VMDCAVE
  if (which_display == DISPLAY_CAVE) {  // call the CAVE specific exit
    CAVEExit();
  }
#endif
  exit(exitcode);
}


// exit the program abnormally; print out special message
void VMDdie(char *exitmsg) {
  msgErr << "Subroutine DIE called ..." << sendmsg;
  if(exitmsg)
    msgErr << exitmsg << sendmsg;
  VMDexit("Program terminating due to fatal error.", 1);
}

