/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeFileGraph.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 1995/05/11 23:18:12 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *  read in a simple format for drawing a colored 2d scalar grid
 *
 ***************************************************************************/

/////// read in a file in the format
/// minx maxx miny maxy nx ny
/// nx*ny data points
// (just skips whitespace, so "minx maxx miny maxy nx ny data0 data1 ... works)
// and draw the graph with each data point being a square
//  The color is (blue -> black -> red) with the most negative number
// being full blue, 0 is black, and the most positive is full red (0,0,1)
// Features have better be added to make this cool.

#include <stdio.h>
#include "MoleculeFileGraph.h"
#include "DispCmds.h"
#include "Inform.h"

MoleculeFileGraph::MoleculeFileGraph(char *filename, Scene *sc)
: MoleculeFile(filename, sc)
{
   grid = NULL;
}
MoleculeFileGraph::MoleculeFileGraph(char *filename, Displayable *disp)
: MoleculeFile(filename, disp)
{
   grid = NULL;
}

MoleculeFileGraph::~MoleculeFileGraph(void)
{
   if (grid) {
      delete grid;
   }
}

int MoleculeFileGraph::create(void)
{
   grid = NULL;
   FILE *infile = fopen(strFile, "r");
   
   if (!infile) {
      msgErr << "Cannot open graph file " << strFile << sendmsg;
      return FALSE;
   }

   int nx, ny;
   minz = -1;
   maxz = 1;
   int n = fscanf(infile, "%f %f %f %f %d %d", &minx, &maxx, &miny, &maxy,
		  &nx, &ny);
   if (n!=6) { // didn't read everything
      msgErr << "Cannot read header (minx maxx miny maxy nx ny) for ";
      msgErr << strFile << "\n";
      fclose(infile);
      return FALSE;
   }
   if (minx == 0.0 && maxx == 0.0) {
      maxx = 1.0;
   }
   if (miny == 0.0 && maxy == 0.0) {
      maxy = 1.0;
   }
   grid = new Grid<float>(nx, ny, 1);
   if (!grid) {
      msgErr << "Cannot create Grid for MoleculeFileGraph" << sendmsg;
      fclose(infile);
      return FALSE;
   }
   int i,j;
   float temp;
   for (i=0; i<nx; i++) {
      for (j=0; j<ny; j++) {
	 if (fscanf(infile, "%f", &((*grid)[i][j][1])) != 1) {
	    msgErr << "Not enough data (looking for " << nx << " * ";
	    msgErr << ny << " = " << nx*ny << " elements)." << sendmsg;
	    delete grid;
	    grid = NULL;
	    fclose(infile);
	    return FALSE;
	 }
	 temp = (*grid)[i][j][1];
	 if (i==0 && j==0) {
	    mindata = maxdata = temp;
	 }
	 if (mindata > temp) {
	    mindata = temp;
	 }
	 if (maxdata < temp) {
	    maxdata = temp;
	 }
      }
   }
   if (fscanf(infile, "%f", &temp) == 1) {
      msgWarn << "More than " << nx << " * " << ny << "(= " << nx*ny << ")";
      msgWarn << "elements" << sendmsg;
      msgWarn << "But I'll take it anyway" << sendmsg;
   }

   fclose(infile);
   return(MoleculeFile::create());  // doesn't really do anything for this data
}

void MoleculeFileGraph::create_cmdlist(void)
{
   reset_disp_list();
   if (!grid) {
      msgErr << "No grid information to graph!" << sendmsg;
      return;
   }
   int nx, ny, nz;
   grid->numrange(&nx, &ny, &nz);
   float pos1[3];
   float pos2[3];
   float pos3[3];
   float dx = (maxx - minx) / nx;
   float dy = (maxy - miny) / ny;
   
   pos1[2] = pos2[2] = pos3[2] = 0.0;
   DispCmdSquare sq;
   DispCmdColorRGB col;
   float rgb[3];
   rgb[1] = 0.0;
   float temp;
   float rscale = maxdata, bscale = mindata;
   if (rscale == 0.0) {
      rscale = 1.0;
   }
   if (bscale == 0.0) {
      bscale = -1.0;
   }
   for (int i=0; i<nx; i++) {
      for (int j=0; j<ny; j++) {
	 pos1[0] = i*dx - dx/2.0 + minx;
	 pos1[1] = j*dy - dy/2.0 + miny;
	 pos2[0] = pos1[0] + dx;
	 pos2[1] = pos1[1];
	 pos3[0] = pos2[0];
	 pos3[1] = pos1[1] + dy;
	 if ((temp = (*grid)[i][j][1])<0) {
	    rgb[0] = 0.0;
	    rgb[2] = temp / bscale;
	 } else {
	    rgb[0] = temp / rscale;
	    rgb[2] = 0.0;
//	    msgInfo << "Put " << rgb[0] << "\n" << sendmsg;
	 }
	 col.putdata(rgb, this);
	 sq.putdata(pos1, pos2, pos3, this);
      }
   }
//   msgInfo << "Done" << sendmsg;
}


// functions to "normalize" the picture  -- for a centering
// translation and a scaling
void MoleculeFileGraph::cov(float &x, float &y, float &z) {
   x = (minx + maxx)/2;
   y = (miny + maxy)/2;
   z = (minz + maxz)/2;
}

float MoleculeFileGraph::scale_factor(void) {
   float wx = maxx - minx;
   float wy = maxy - miny;
   float wz = maxz - minz;
   if (wx > wy) {
      if (wx > wz) {
	 return 2.0/wx;
      } else {
	 return 2.0/wz;
      }
   } else {
      if (wy > wz) {
	 return 2.0/wy;
      } else {
	 return 2.0/wz;
      }
   }
}

