/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: MoleculeFileRaster3D.C,v $
 *	$Author: khamer $	$Locker:  $		$State: Exp $
 *	$Revision: 1.5 $	$Date: 1995/05/12 19:16:33 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *   read in from the Raster3D file format
 *
 ***************************************************************************/

////  The header contains a lot of information which I ignore
// TITLE (80 chars)
// NTX, NTY (number of tiles -- ignored)
// NPX, NPY (number of points per tile -- ignored)
// SCHEME (antialiasing scheme -- ignored)
// BKGND (background color -- ignored)
// SHADOW (T/F -- ignored)
// IPHONG (phong power -- ignored)
// STRAIT (secondary light source -- ignored)
// AMBIEN (ambient illumination -- ignored)
// SPECLR ignored
// EYEPOS ignored
// SOURCE ignored
// TMAT 4x4 matrix used to view the system
// object mode (1 == triangle, 2 == spheres, or 3 ==  mixed)
// INFMTS  --  FORTRAN input field specifications
//   for input format specifiers 1/6, 2, and 3/5
//   triangle(1) or plane(6) (x1,y1,z1)-(x2,y2,z2)-(x3,y3,z3) (r,g,b)
//   sphere(2)      (x,y,z) r (r,g,b)
//   cylinder(3,5)  (x1,y1,z1) R1 - (x2,y2,z2) R2(is ignored) (r,g,b)
//   I ignore these lines and just read them in as "%f"

#include <stdio.h>
#include "MoleculeFileRaster3D.h"
#include "DispCmds.h"
#include "Inform.h"
#include "ColorList.h"
#include "Global.h"


MoleculeFileRaster3D::MoleculeFileRaster3D(char *filename, Scene *sc)
: MoleculeFile(filename, sc)
{
}
MoleculeFileRaster3D::MoleculeFileRaster3D(char *filename, Displayable *disp)
: MoleculeFile(filename, disp)
{
}

MoleculeFileRaster3D::~MoleculeFileRaster3D(void)
{
}

// read (x,y,z)[0], (x,y,z)[1], (x,y,z)[2], (r,g,b)
static int get_triangle(FILE *infile, float *arr)
{
   if( fscanf(infile, "%f %f %f %f %f %f %f %f %f %f %f %f\n",
	      arr  , arr+1, arr+2,
	      arr+3, arr+4, arr+5,
	      arr+6, arr+7, arr+8,
	      arr+9, arr+10, arr+11) < 12) { 
      msgErr << "Could not read a triangle -- skipping" << sendmsg;
      return FALSE;
   }
   return TRUE;
}

// read (x,y,z), R, (r,g,b)
static int get_sphere(FILE *infile, float *arr)
{
   if (fscanf(infile, "%f %f %f %f %f %f %f\n",
		 arr, arr+1, arr+2,
		 arr+3,
		 arr+4, arr+5, arr+6) < 7) {
      msgErr << "Could not read a sphere -- skipping" << sendmsg;
      return FALSE;
   }
   return TRUE;
}

// read (x,y,z)[0], R[0], (x,y,z)[1], R[1], (r,g,b)
static int get_cylinder(FILE *infile, float *arr)
{
   if (fscanf(infile, "%f %f %f %f %f %f %f %f %f %f %f\n",
		 arr  , arr+1, arr+2, arr+3,
		 arr+4, arr+5, arr+6, arr+7,
		 arr+8, arr+9, arr+10) < 10) {
      msgErr << "Could not read a cylinder -- skipping" << sendmsg;
      return FALSE;
   }
   return TRUE;
}

// read the file and put the info into the drawing list
int MoleculeFileRaster3D::create(void)
{
   FILE *infile = fopen(strFile, "r");
   if (!infile) {
      msgErr << "Cannot open Raster3D file " << strFile << sendmsg;
      return FALSE;
   }

   Matrix4 mat;
   char header[100];
   if (!fgets(header, 99, infile)) {
      fclose(infile);
      return FALSE;
   }
   msgInfo << header << sendmsg;  // tell the user info about the file

   int count = 11;    // ignore the next 11 lines of text
   char buffer[200];
   while (count--) {
      if (!fgets(buffer, 99, infile)) {
	 fclose(infile);
	 msgErr << "Could not read Raster3D file (" << strFile << ")";
	 msgErr << " header information." << sendmsg;
	 return FALSE;
      }
   }
   // Now I have to get the matrix.  This is made nasty since
   // there could be extra text after the first four numbers on a line
   // as in: 1 0 0 This is an extra comment
   for (int i=0; i<4; i++) {
      int j;
      fgets(buffer, 199, infile);  // read the whole line into a string
      if (j=sscanf(buffer, "%f %f %f %f",
		 &mat.mat[i][0], &mat.mat[i][1], &mat.mat[i][2],
		 &mat.mat[i][3])<4) {
//msgErr << "i= " << i << "  j= " << j << " info: " << buffer << sendmsg;
	 msgErr << "Could not read all of " << strFile << " line ";
	 msgErr << i+1 << " of the Raster3D viewing matrix." << sendmsg;
	 fclose(infile);
	 return FALSE;
      }
//      msgInfo << "matrix: " << buffer << sendmsg;
   }
   int inputtype;  // get the input type (1, 2, or 3)
   fgets(buffer, 199, infile);
   if (sscanf(buffer, "%d", &inputtype) < 1) {
      msgErr << "Could not read the object mode line in " << strFile;
      msgErr << " Raster3D file." << sendmsg;
      fclose(infile);
      return FALSE;
   }
   if (inputtype < 1 || inputtype > 3) {
      msgErr << "Unknown mode type " << inputtype << " in ";
      msgErr << strFile << sendmsg;
      fclose(infile);
      return FALSE;
   }
   
   // do top-level creation; remember return code
   int createRetVal = MoleculeFile::create();
   if(!createRetVal) {
     // error doing top-level create ... abort
     fclose(infile);
     return FALSE;
   }

   // since we only need to do this once, create display list here
   reset_disp_list();

//   DispCmdMult multmat(&mat);
//   multmat.put( this);
   DispCmdColorIndex col;
   DispCmdMaterials materials;
   DispCmdTriangle tri;
   DispCmdSphere sph;
   DispCmdCylinder cyl;
   float data[15];
   count=0;
   int triangle_count = 0; // this is a hack to try and
   // get the triagulation for the ribbons from molscript, etc. to
   // look right.  I alternate ribbons front and back

   materials.putdata(TRUE, this);
   // note, I also have to get the min/max values to find the 
   // cov and scale parameters
   if (inputtype == 1) { // all triangles  -- not yet tested
      fgets(buffer, 199, infile); // ignore the input specifier
      while (!feof(infile)) {
	 if (get_triangle(infile, data) ) {
	    col.putdata(::colors -> nearest_index(
		data[9], data[10], data[11]), this);
	    if (!triangle_count) {
	       tri.putdata(data, data+3, data+6, this);
	       triangle_count = 1;
	    } else {
	       tri.putdata(data, data+6, data+3, this);
	       triangle_count = 0;
	    }
	    if (count++ == 0) {
	       minx = maxx = data[0];
	       miny = maxy = data[1];
	       minz = maxz = data[2];
	    }
	    for (i=0; i<3; i++) {
	       if (data[3*i+0] < minx) minx = data[3*i+0];
	       if (data[3*i+1] < miny) miny = data[3*i+1];
	       if (data[3*i+2] < minz) minz = data[3*i+2];
	       if (data[3*i+0] > maxx) maxx = data[3*i+0];
	       if (data[3*i+1] > maxy) maxy = data[3*i+1];
	       if (data[3*i+2] > maxz) maxz = data[3*i+2];
	    }
	 }
      }
   } else if (inputtype == 2) { // all spheres  -- not yet tested
      fgets(buffer, 199, infile); // ignore the input specifier
      while (!feof(infile)) {
	 if (get_sphere(infile, data) ) {
	    col.putdata(::colors -> nearest_index(
		data[4], data[5], data[6]), this);
	    sph.putdata(data, data[3], this);
	    if (count++ == 0) {
	       minx = maxx = data[0];
	       miny = maxy = data[1];
	       minz = maxz = data[2];
	    }
	    for (i=0; i<1; i++) {
	       if (data[3*i+0] < minx) minx = data[3*i+0];
	       if (data[3*i+1] < miny) miny = data[3*i+1];
	       if (data[3*i+2] < minz) minz = data[3*i+2];
	       if (data[3*i+0] > maxx) maxx = data[3*i+0];
	       if (data[3*i+1] > maxy) maxy = data[3*i+1];
	       if (data[3*i+2] > maxz) maxz = data[3*i+2];
	    }
	 }
      }
   } else { // mixed
      fgets(buffer, 199, infile); // ignore the input specifiers
      fgets(buffer, 199, infile);
      fgets(buffer, 199, infile);
      while (!feof(infile)) {
	 int objtype = -1;
	 int error_check; 
	 if (!fgets(buffer, 199, infile)) {
	    continue;
	 }
	 // I do some checking in case things go whacko and a line
	 // was dropped, for whatever reasons.  This will attempt
	 // to get back to a line w/ only one item
	 if (sscanf(buffer, "%d %d", &objtype, &error_check)==1) {
	    switch(objtype) {
	     case 1:
	       if (get_triangle(infile, data)) {
		  col.putdata(::colors -> nearest_index(
			data[9], data[10], data[11]), this);
  // flip every other triangle to see ribbons correctly
  // otherwise I get a fron/ back pattern.  This is a hack.
  // Should I try telling vmd to use two-sided polygons?
		  if (!triangle_count) {
		     tri.putdata(data, data+3, data+6, this);
		     triangle_count = 1;
		  } else {
		     tri.putdata(data, data+6, data+3, this);
		     triangle_count = 0;
		  }
		  if (count++ == 0) {
		     minx = maxx = data[0];
		     miny = maxy = data[1];
		     minz = maxz = data[2];
		  }
		  for (i=0; i<3; i++) {
		     if (data[3*i+0] < minx) minx = data[3*i+0];
		     if (data[3*i+1] < miny) miny = data[3*i+1];
		     if (data[3*i+2] < minz) minz = data[3*i+2];
		     if (data[3*i+0] > maxx) maxx = data[3*i+0];
		     if (data[3*i+1] > maxy) maxy = data[3*i+1];
		     if (data[3*i+2] > maxz) maxz = data[3*i+2];
		  }
	       }
	       break;
	     case 2: 
	       triangle_count = 0;
	       if (get_sphere(infile, data) ) {  
		  col.putdata(::colors->nearest_index(
			data[4], data[5], data[6]), this);
		  sph.putdata(data, data[3], this);
		  if (count++ == 0) {
		     minx = maxx = data[0];
		     miny = maxy = data[1];
		     minz = maxz = data[2];
		  }
		  for (i=0; i<1; i++) {
		     if (data[3*i+0] < minx) minx = data[3*i+0];
		     if (data[3*i+1] < miny) miny = data[3*i+1];
		     if (data[3*i+2] < minz) minz = data[3*i+2];
		     if (data[3*i+0] > maxx) maxx = data[3*i+0];
		     if (data[3*i+1] > maxy) maxy = data[3*i+1];
		     if (data[3*i+2] > maxz) maxz = data[3*i+2];
		  }
	       }
	       break;
	     case 3:  // these are the same so far
	     case 5:
	       triangle_count = 0;
	       if (get_cylinder(infile, data)) {
		  col.putdata(::colors->nearest_index(
			data[8], data[9], data[10]), this);
		  cyl.putdata(data, data+4, data[3], 8, this);
		  // spheres removed for speed
//		  sph.putdata(data, data[3], this);
//		  sph.putdata(data+4, data[3], this);
		  if (count++ == 0) {
		     minx = maxx = data[0];
		     miny = maxy = data[1];
		     minz = maxz = data[2];
		  }
		  for (i=0; i<2; i++) {
		     if (data[3*i+0+i] < minx) minx = data[3*i+0+i];
		     if (data[3*i+1+i] < miny) miny = data[3*i+1+i];
		     if (data[3*i+2+i] < minz) minz = data[3*i+2+i];
		     if (data[3*i+0+i] > maxx) maxx = data[3*i+0+i];
		     if (data[3*i+1+i] > maxy) maxy = data[3*i+1+i];
		     if (data[3*i+2+i] > maxz) maxz = data[3*i+2+i];
		  }
	       }
	       break;
	     default:{
	       triangle_count = 0;
		msgErr << "I don't (yet?) understand object type ";
		msgErr << objtype << " -- skipping it" << sendmsg;
		msgErr << "skipped: " << buffer << sendmsg;
		fgets(buffer, 199, infile);
		msgErr << "skipped: " << buffer << sendmsg;
		break;
	     }
	    } // end of switch
	 }  else { // end if "got an input"
	    msgErr << "Expected one element on the line, got: "<< sendmsg;
	    msgErr << " " << buffer << sendmsg;
	 }
      } // end of while
   } // end of type "mixed objects"
   materials.putdata(FALSE, this);
   fclose(infile);
   return createRetVal;
} // end of function


// center of volume
void MoleculeFileRaster3D::cov(float &x, float &y, float&z) {
   x = (minx + maxx)/2;
   y = (miny + maxy)/2;
   z = (minz + maxz)/2;
}

// get the size
float MoleculeFileRaster3D::scale_factor(void) {
   float wx = maxx - minx;
   float wy = maxy - miny;
   float wz = maxz - minz;
   if (wx > wy) {
      if (wx > wz) {
	 return 2.0/wx;  // the 2.0 is to fit the system in
      } else {           // a box from -1 to 1
	 return 2.0/wz;
      }
   } else {
      if (wy > wz) {
	 return 2.0/wy;
      } else {
	 return 2.0/wz;
      }
   }
}

