/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *      $RCSfile: PickModeMolLabel.C,v $
 *      $Author: billh $        $Locker:  $                $State: Exp $
 *      $Revision: 1.3 $      $Date: 1995/05/11 23:31:23 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The PickMode object which allows a pointer to be used to create new 
 * geometrical monitoring labels.  This particular version is used only for
 * adding molecular labels, i.e. Atoms, Bonds, Angles, and Dihedrals.  As
 * more atoms are selected, they are remembered until enough have been picked
 * to create the relevant label (i.e. 3 atoms to create an Angle label).
 *
 ***************************************************************************/

#include <math.h>
#include "PickModeMolLabel.h"
#include "Pickable.h"
#include "DisplayDevice.h"
#include "Inform.h"
#include "MoleculeList.h"
#include "Molecule.h"
#include "CommandQueue.h"
#include "CmdLabel.h"
#include "Global.h"
#include "utilities.h"

// typedef used for creating array of Molecule pointers
typedef Molecule *molObjPtr;
typedef char *molNamePtr;

// static flag for this class
int PickModeMolLabel::molPickModeActive = 0;
int PickModeMolLabel::molPickModeNextID = 1;

/////////////////////////////  constructor  ///////////////////////////
PickModeMolLabel::PickModeMolLabel(char *nm, int size) {

  // get ID of this object
  id = molPickModeNextID++;

  // save our name
  modeName = stringdup(nm);

  // save number of elements needed, and allocate storage
  needItems = size;
  if(size > 0) {
    molData = new molObjPtr[size];
    atomData = new molNamePtr[size];
    for(int i=0; i < size; i++)
      atomData[i] = new char[128];
  } else {
    molData = NULL;
    atomData = NULL;
  }

  // indicate we're still at the starting trying to find something
  haveItems = 0;
  pQuery = NULL;

  MSGDEBUG(1,"Created new PickModeMolLabel " << modeName << " needing ");
  MSGDEBUG(1,needItems << " items." << sendmsg);
}

///////////////////////////////  destructor  /////////////////////////
PickModeMolLabel::~PickModeMolLabel(void) {

  // free up space if necessary
  if(needItems > 0) {
    for(int i=0; i < needItems; i++)
      delete [] (atomData[i]);
    delete [] atomData;
    delete [] molData;
  }

  delete [] modeName;

  // if we're currently trying to create a monitor, give up
  if(molPickModeActive == id)
    molPickModeActive = 0;
}


/////////////////////////////  private routines  //////////////////////

// For the given Pickable, determine if it is a molecule or a representation
// of a molecule, and return the proper pointer if it is (NULL otherwise)
Molecule *PickModeMolLabel::check_pickable(Pickable *pobj) {
  register int i,j, mnum, repnum;

  if(!moleculeList)
    return NULL;

  // search through each molecule in the MoleculeList ...
  mnum = moleculeList->num();
  for(i=0; i < mnum; i++) {

    // and check each molecule's representation to see if it matches
    Molecule *mol = moleculeList->molecule(i);
    if(pobj == mol)
      return mol;

    repnum = mol->components();
    for(j=0; j < repnum; j++)
      if(pobj == mol->component(j))
	return mol;
  }

  // if here, nothing found
  return NULL;
}


///////////////////////////  public virtual routines  /////////////////////

// called when a pick is begun:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// Here, who cares what the button was
void PickModeMolLabel::pick_start(DisplayDevice *,
			Pickable *p, int, int, int dim, float *pos) {

  // have we selected a molecule?
  if((pQuery = check_pickable(p)) != NULL) {

    // if we're not starting out, but the mode has changed, reset to start
    if(haveItems != 0 && molPickModeActive != id)
      haveItems = 0;

    // initialize at start of selecting atom
    memcpy((void *)pPos, (void *)pos, dim*sizeof(float));
    molPickModeActive = id;
    needName = TRUE;
  }

}

  
// called when a pick moves:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// Here, who cares what the button was
void PickModeMolLabel::pick_move(DisplayDevice *,
			Pickable *, int, int, int dim, float *pos) {

  // just keep track to see if the pointer moves any; if so, cancel action
  if(needName) {
    float mvdist = 0.0;
    for(int i=0; i < dim; i++)
      mvdist += fabs(pPos[i] - pos[i]);

    if(mvdist > 0.01)
      needName = FALSE;
  }
}


// called when a pick ends:
//	args = display to use, obj picked, button, tag, dim, pos
// For 2D version: x & y are 0 ... 1, represent 'relative, scaled' coords.
// For 3D version: x,y,z are transformed position of pointer
// Here, who cares what the button was
void PickModeMolLabel::pick_end(DisplayDevice *,
			Pickable *, int, int atom, int, float *) {

  if(needName) {
    // the selection was successful; first save the info for the object
    molData[haveItems] = pQuery;
    pQuery->atom_full_name(atom, atomData[haveItems]);

    // every time an atom is selected, add an Atoms label
    commandQueue->append(new CmdLabelAdd("Atoms", 1, atomData + haveItems));

    // indicate we have one more items in the ones we need
    haveItems++;

    // now check if we have enough items for the object we're out for
    if(haveItems >= needItems) {

      if(needItems > 1) {
        // for labels other than just atoms, add new monitor
        // PUT CHECK HERE WHETHER IT EXISTS OR NOT; IF EXISTS, DELETE INSTEAD
        commandQueue->append(new CmdLabelAdd(modeName, needItems, atomData));
      }

      // indicate we're done with this selection
      haveItems = 0;
      molPickModeActive = 0;
    }
  }
  pQuery = NULL;
  needName = FALSE;
}

