/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: PopupMenu.h,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 1995/06/07 03:53:33 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * The pop-up (possibly pull-down) menu used by the Mouse and perhaps other
 * components.  The menu consists of elements with these characteristics:
 *	1) Name (as it appears in the menu)
 *	2) Text command to execute when the item is selected (may be null).
 *	3) Return code: this is an integer code which is returned by the
 *	   windowing system when the item is picked.   It is assigned by the
 *	   pop-up menu object, and for all regular menu items, it is > 0.
 *	   There are several "special" return codes, which indicate the menu
 *	   item is special.  These special return codes include:
 *	   	MENU_CODE_SUBMENU: the menu item contains a submenu
 *		MENU_CODE_SEP: the menu item is really a separator
 *		MENU_CODE_NONE: the menu item cannot be selected, it just
 *			exists to display text
 *	4) submenu object, if any
 *	5) flags to indicate if the menu item should be displayed, or if
 *	   it should be disabled (greyed out) if possible.
 *
 * A pop-up menu relys on a specific windowing system to provide the methods
 * for creating, posting, and activating a menu.
 *
 * A pop-up menu contains a list of pop-up items, each item containing the
 * characteristics listed above.  A menu is a list of these, and has a name
 * associated with the total list as well as with each item.
 *
 * These classes are used in the following way:
 *	1) A PopupMenu instance is created, that is at the start empty
 *	2) The user calls PopupMenu->add_item to add commands, until the
 *		menu is defined.  This will return integer "return codes"
 *		for the items added, which must be remembered by the caller.
 *	3) If there is a heirarchy of menus, create submenus as described,
 *		and use add_submenu to the parent menus.
 *	4) When the menu is defined, and it is to be used, the menu creator
 *		calls PopupMenu->activate(), which returns either a pointer
 *		to a PopupMenuItem, or NULL if nothing was selected.  The
 *		caller can use this item to determine the return code and
 *		the text command.
 ***************************************************************************/
#ifndef POPUP_OBJ_H
#define POPUP_OBJ_H

#include "NameList.h"
#include "utilities.h"
class Inform;
class DisplayDevice;


// enumeration describing the types of return codes (these are all <= 0;
// returns codes > 0 are assigned dynamically)
enum PopupCodes { 	MENU_CODE_NONE = 0,	MENU_CODE_SUBMENU = (-1),
			MENU_CODE_SEP = (-2) };

/////////////////////////  forward declarations
class PopupMenuItem;
class PopupMenu;


/////////////////////////  pop-up menu item class definition

class PopupMenuItem {

private:
  // return code for this item
  int returnCode;

  // text string command for this item
  char *textCmd;

  // name of this item (as it should appear in the menu)
  char *itemName;

  // pointer if this contains a submenu
  PopupMenu *subMenu;

  // flags for whether this item should be enabled/disabled,
  // or should have a check mark next to it
  int showItem, showCheck, checkItem;

public:
  // constructor 1: specify return code, name, and text command
  // (the text command may be NULL, in which case no action is done if this
  //  item is selected)
  PopupMenuItem(int, char *, char * = NULL);

  // constructor 2: submenu (which determines the name and return code)
  PopupMenuItem(PopupMenu *);

  // constructor 3: nothing ... a separator
  PopupMenuItem(void);

  // destructor: frees up allocated memory
  ~PopupMenuItem(void);

  //
  // return info about the item
  //

  int is_separator(void) { return returnCode == MENU_CODE_SEP; }
  int is_submenu(void) { return returnCode == MENU_CODE_SUBMENU; }

  int return_code(void) { return returnCode; }
  int showing(void) { return showItem; }
  int show_check(void) { return showCheck; }
  int checked(void) { return checkItem && showCheck; }

  char *text_command(void) { return textCmd; }
  char *name(void) { return itemName; }
  PopupMenu *sub_menu(void) { return subMenu; }

  // print out the contents of the menu to the given Inform object
  // args: Inform object to print to, and leading spaces for each line
  int print(Inform *, int = 0);

  //
  // change status of item
  //

  void return_code(int nc) { returnCode = nc; }
  void showing(int yn) { showItem = yn; }
  void show_check(int yn) { showCheck = yn; }
  void checked(int yn) { checkItem = yn; }

};


/////////////////////////  pop-up menu class definition

class PopupMenu {

public:
  // integer id for this menu; may not be necessary for windowing system
  long menuID;

  // flag that may be used by windowing system to indicate if menu needs
  // to be deallocated/deleted/whatever
  int menuExists;

private:
  // static integer indicating what the next available return code is.
  static int nextReturnCode;

  // array with menu items
  ResizeArray<PopupMenuItem *> menuItems;

  // name of this menu
  char *menuName;

public:
  // constructor: name of menu, and initial guess on how many items it contain
  PopupMenu(char *, int = 8);

  // destructor: deletes all popup items, which may include submenus as well
  ~PopupMenu(void);

  //
  // return info about the menu
  //

  char *name(void) { return menuName; }
  int items(void);

  // return the Nth item, or NULL if illegal N given
  PopupMenuItem *menu_item(int);

  // return the menu item with the given label or NULL if not found
  PopupMenuItem *menu_item(char *);

  // given a return code, searches the menu until the first item with that
  // code is found.  Returns the item pointer, or else NULL if none are found
  // or the given return code is <= 0.
  // The 2nd argument is used to determine where to start, if it is NULL, the
  // current menu is used.  This is done so this routine can be recursive.
  PopupMenuItem *find_item(int, PopupMenu * = NULL);

  // print out the contents of the menu to the given Inform object
  // args: Inform object to print to, and leading spaces for each line
  int print(Inform *, int = 0);

  //
  // add new items to the menu
  //

  // add a regular item; return the return code for the item
  // arguments: item name, text command, whether to display it or not,
  // whether to show a check mark, and if a check mark, whether the check
  // is on or off
  int add_item(char *, char *, int, int, int = FALSE);

  // add a separator; return success
  int add_separator(void);

  // add a submenu; return success
  // arguments: menu, whether to display it or not,
  // whether to show a check mark, and if a check mark, whether the check
  // is on or off
  int add_submenu(PopupMenu *, int, int, int = FALSE);

  // delete the Nth item.  Return success.
  int delete_item(int);

  //
  // activate the pop-up menu
  //

  // request the pop-up menu be activated by the windowing system; this
  // will either return NULL, or the defined item.
  PopupMenuItem *activate(DisplayDevice *);

};

#endif

/***************************************************************************
 * REVISION HISTORY:
 *
 * $Log: PopupMenu.h,v $
 * Revision 1.3  1995/06/07  03:53:33  billh
 * Moved final endif to very end of file.
 *
 * Revision 1.2  95/05/12  05:56:40  billh
 * Removed extern class definition.
 * 
 * Revision 1.1  95/05/11  23:20:04  billh
 * Initial revision
 * 
 ***************************************************************************/
