/***************************************************************************
 *cr                                                                       
 *cr            (C) Copyright 1995 The Board of Trustees of the           
 *cr                        University of Illinois                       
 *cr                         All Rights Reserved                        
 *cr                                                                   
 ***************************************************************************/

/***************************************************************************
 * RCS INFORMATION:
 *
 *	$RCSfile: ReadPSF.C,v $
 *	$Author: billh $	$Locker:  $		$State: Exp $
 *	$Revision: 1.3 $	$Date: 1995/05/11 23:42:30 $
 *
 ***************************************************************************
 * DESCRIPTION:
 *
 * Routines to read .psf files.
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ReadPSF.h"


/* Open the .psf file and skip past the remarks to the first data section.
   Returns the file pointer, or NULL if error.  Also puts the number of
   atoms in the molecule into the given integer.  If fname is an empty string,
   this will use standard input. */
FILE *open_psf_file(char *fname, int *natom) {
  
  char inbuf[PSF_RECORD_LENGTH+2];
  FILE *f;
  
  /* open the file; if error, return NULL	*/
  if(strlen(fname) == 0) {
    f = stdin;
  } else {
    if(!(f = fopen(fname,"r"))) {
      *natom = 0;
      return NULL;
    }
  }

  *natom = 0;
      
  /* keep reading the next line until a line with NATOM and without 
     REMARKS appears	*/
  do {
    if(inbuf != fgets(inbuf, PSF_RECORD_LENGTH+1, f)) {
      /* EOF encountered with no NATOM line found ==> error, return null */
      *natom = 0;
      fclose(f);
      return NULL;
    }
    if(strlen(inbuf) > 0) {
      if(!strstr(inbuf,"REMARKS")) {
        if(strstr(inbuf,"NATOM")) {
          *natom = atoi(inbuf);
        }
      } else {
        fputs(inbuf,stderr);
      }
    }
  } while (*natom == 0);
  
  return f;
}


/* Read in the next atom info into the given storage areas; this assumes
   that file has already been moved to the beginning of the atom records.
   Returns the serial number of the atom. If there is an error, returns -1.*/
int get_psf_atom(FILE *f, char *name, char *atype, char *resname, 
   char *segname, int *resid, float *q, float *m) {
   
  char inbuf[PSF_RECORD_LENGTH+2];
  int i,num;

  if(inbuf != fgets(inbuf, PSF_RECORD_LENGTH+1, f)) {
    return(-1);
  }
    
  num = atoi(inbuf);
  
  strncpy(segname,inbuf+9,4); segname[4] = '\0';
  strncpy(resname,inbuf+19,4);  resname[4] = '\0';
  strncpy(name,inbuf+24,4);  name[4] = '\0';
  strncpy(atype,inbuf+29,4); atype[4] = '\0';

  for(i=3; i >= 0; i--) {
    if(segname[i] == ' ')	segname[i] = '\0';
    if(resname[i] == ' ')	resname[i] = '\0';
    if(name[i] == ' ')		name[i] = '\0';
    if(atype[i] == ' ')		atype[i] = '\0';
  }    

  *resid = atoi(inbuf+13);
  *q = atof(inbuf+34);
  *m = atof(inbuf+50);
  
  if(num < 5) {
    fprintf(stderr,"Atom:%-6d",num);
    fprintf(stderr," SN:%4s",segname);
    fprintf(stderr," ID:%5d",*resid);
    fprintf(stderr," RN:%4s",resname);
    fprintf(stderr," NM:%4s",name);
    fprintf(stderr," TP:%4s",atype);
    fprintf(stderr," Q=%+5.3f M=%+6.3f\n",*q,*m);
  }
  
  return num;
}


/* Read in the beginning of the bond information, but don't read in the 
   bonds.  Returns the number of bonds in molecule.  If error, returns (-1). */
int start_psf_bonds(FILE *f) {

  char inbuf[PSF_RECORD_LENGTH+2];
  int nbond = 0;
  
  /* keep reading the next line until a line with NBOND appears */
  do {
    if(inbuf != fgets(inbuf, PSF_RECORD_LENGTH+1, f)) {
      /* EOF encountered with no NBOND line found ==> error, return (-1) */
      return (-1);
    }
    if(strlen(inbuf) > 0 && strstr(inbuf,"NBOND"))
      nbond = atoi(inbuf);
  } while (nbond == 0);

  return nbond;
}


/* Read in the bond info into the given integer arrays, one for 'from' and
   one for 'to' atoms; remember that .psf files use 1-based indices,
   not 0-based.  Returns TRUE if all nbond bonds found; FALSE otherwise.  */
int get_psf_bonds(FILE *f, int nbond, int fromAtom[], int toAtom[]) {

  char *bondptr, inbuf[PSF_RECORD_LENGTH+2];
  int i=0;

  while(i < nbond) {
    if((i % 4) == 0) {
      /* must read next line */
      if(!fgets(inbuf,PSF_RECORD_LENGTH+2,f)) {
        /* early EOF encountered */
        break;
      }
      bondptr = inbuf;
    }
    if((fromAtom[i] = atoi(bondptr)) < 1)
      break;
    bondptr += 8;
    if((toAtom[i] = atoi(bondptr)) < 1)
      break;
    bondptr += 8;
    i++;
  }
  
  return (i == nbond);
}

