#!/usr/bin/awk -f
# Umbrella estimator based on 2D histogram from the colvars.traj file
# Jerome Henin, 2012

# Parameters are passed with -v param=value

# Krestr (mandatory: -v Krestr=XXX), get value from colvars output
# "Computed extended system force constant"

# bin widths dx and dy:
# x is the cv value
# y is the ref. (restraining) value = extended coord

# if different from 300 K, set the temp variable


function floor(y){
  if (y < 0){
    i = int(y)
    if(i == y)
      return i
    else
      return i-1
  }
  else 
    return int(y)
}

function write_pmf() {
# implements eq. (5) from Harris et al. 2012 (?) dynamic ref. restraining
  # first get xi_av(lambda) i.e. x_av(y), and std. dev.
  for (biny = miny; biny <= maxy; biny++) {
    #s = 0
    #for (binx = minx; binx <= maxx; binx++) {
    #  norm += count[binx,biny]
    #  s += count[binx,biny] * (binx + 0.5) * dx
    #}
    #x_av[biny] = s / (norm ? norm : 1)

    # code above replaced with simpler:
    norm = county[biny] ? county[biny] : 1
    x_av[biny] = sumx[biny] / norm

    s = 0
    for (binx = minx; binx <= maxx; binx++) {
      x = (binx + 0.5) * dx
      s += count[binx,biny] * (x - x_av[biny])^2
      # output xi histogram
#        print x, count[binx,biny] / (norm ? norm : 1) > "histogram.dat"
    }
#    print ""  > "histogram.dat"
#    sigma2[biny] = s / (norm ? norm : 1)
    sigma2[biny] = sumx2[biny]/norm - x_av[biny]^2
#    print "SIGMA2", biny, x_av[biny], sigma2[biny], sigma2_alt[biny] > "/dev/stderr"
  }

  for (binx = minx; binx <= maxx; binx++) {
    x = (binx + 0.5) * dx
    norm = 0
    av = 0
    diff_av = 0
    for (biny = miny; biny <= maxy; biny++) {
      y = (biny + 0.5) * dy
      norm += count[binx,biny]
      if ( sigma2[biny] ) {
        av += count[binx,biny] * (x - x_av[biny]) / sigma2[biny]
      } else if (count[binx,biny]>1) {
        print "WARNING: non-empty bin gave zero std. dev. (bins too wide?)" > "/dev/stderr"
        print "x = ", x, " y = ", y, "dy =", dy, "Ny = ", county[biny] > "/dev/stderr"
      }
      diff_av += count[binx,biny] * (x - y)
    }
    # For checking: average force (i.e. av. diff)
    diff_av /= (norm ? norm : 1)
    print x, diff_av > "average_diff.dat"
    
    av = BOLTZMANN * temp * av / (norm ? norm : 1)
    printf "%7.3f  %7.3f\n", x, av - Krestr * diff_av
  }
  print ""
}

BEGIN {
    if (!dx) dx = 0.1
    if (!dy) dy = dx

    if (!step) step = -1
    if (!skip) skip = 1

    if (!temp) temp = 300
    BOLTZMANN = 0.00198721
    if ((!width) && (!Krestr)) {
      print "Need to define Krestr!" > "/dev/stderr"
      exit 1
    }
    
    if (!Krestr) {
      Krestr = 0.596157 / (width^2)
      print "Computing Krestr from width (make sure the mass constant is set properly)" > "/dev/stderr"
    }

    print "Krestr =", Krestr > "/dev/stderr"
}

($1 != "#") && (NR % skip == 0) {
    binx = floor($2/dx)
    biny = floor($3/dy)
    # Bin 0 is from 0 to dx or dy
    if (total == 0) {
      maxx = binx
      minx = binx
      maxy = biny
      miny = biny
    } else {
      if (binx > maxx) maxx = binx
      if (binx < minx) minx = binx
      if (biny > maxy) maxy = biny
      if (biny < miny) miny = biny
    }
    count[binx,biny] += 1

    # prepare computation of sigma2 of x for y bin
    sumx[biny] += $2
    sumx2[biny] += $2*$2
    county[biny] += 1
    #margx[binx] += 1
    #margy[biny] += 1
    total++

    if (step > 0 && total % step == 0) {
      write_pmf()
    }
}

END {
    if (step < 0) {
      write_pmf()
    }
}

