
#include "alignedSequence.h"


// Constructor
AlignedSequence::AlignedSequence(int l, Alphabet* a, char* n)
  : Sequence(l,a,n), unalignedLength(0) {
  //printf("=>AlignedSequence\n");
  alignedToUnaligned = new int[getMaxLength()];
  unalignedToAligned = new int[getMaxLength()];
  //printf("<=AlignedSequence\n");

  return;
}


// Constructor
//   arguments:
//      int l - sequence length (including gaps)
//      Alphabet* a - Alphabet for this sequence
AlignedSequence::AlignedSequence(int l, Alphabet* a) 
  : Sequence(l,a), unalignedLength(0) {

  alignedToUnaligned = new int[getMaxLength()];
  unalignedToAligned = new int[getMaxLength()];

  return;
}


// Destructor
AlignedSequence::~AlignedSequence() {

  delete alignedToUnaligned;
  delete unalignedToAligned;

  return;
}


int AlignedSequence::addSymbol(Symbol* symbol) {

  int success = Sequence::addSymbol(symbol);  // increases length
  
  if (success == 0) return success;
  if (unalignedLength == getMaxLength()) return 0;

  alignedToUnaligned[getLength()-1] = unalignedLength;
  unalignedToAligned[unalignedLength] = getLength()-1;
  unalignedLength++;

  return 1;
}


int AlignedSequence::addSymbol(char c) {

  int success = Sequence::addSymbol(c);  // increases length
  
  if (success == 0) return success;
  if (unalignedLength == getMaxLength()) return 0;

  alignedToUnaligned[getLength()-1] = unalignedLength;
  unalignedToAligned[unalignedLength] = getLength()-1;
  unalignedLength++;

  return 1;
}


int AlignedSequence::addGap() {

  //printf("addGap()\n");

  int success = Sequence::addSymbol(getAlphabet()->getSymbol('-'));

  //printf("success: %d\n",success);

  if (success == 0) return success;
  if (unalignedLength <= getMaxLength()) {
    alignedToUnaligned[getLength()-1] = unalignedLength;
  }
  else {
    alignedToUnaligned[getLength()-1] = -1;
  }

  return 1;
}


// getUnalignedSymbol
//
const Symbol* AlignedSequence::getUnalignedSymbol(int i) {

  if (i < unalignedLength &&
      unalignedToAligned[i] >= 0) {
    return getSymbol(unalignedToAligned[i]);
  }

  return 0;
}


// getUnalignedLength
//
int AlignedSequence::getUnalignedLength() {

  return unalignedLength;
}


// alignedToUnalignedIndex
//   return index of non-gapped sequence corresponding to
//   the given gapped sequence index
int AlignedSequence::alignedToUnalignedIndex(int i) {

  if (i < getLength()) {
    return alignedToUnaligned[i];
  }

  return -1;
}


// unalignedToAlignedIndex
//   return index of gapped sequence corresponding to
//   the given non-gapped sequence index
int AlignedSequence::unalignedToAlignedIndex(int i) {

  if (i < unalignedLength) {
    return unalignedToAligned[i];
  }

  return -1;
}

/**
 * This method gets the percentage of elements that this sequence shares with the specified sequence.
 *
 * @param   comparisonSequence  The sequence to compare this sequence to.
 * @return  The percentage of elements that the two sequences have in common or 0.0 on error.
 */
float AlignedSequence::getPercentIdentity(AlignedSequence *comparisonSequence) {

    //Make sure the sequences are of the same length.
    if (getLength() !=  comparisonSequence->getLength()) return 0.0;

    //Go through the symbols and count the ones that are the same.
    int identityCount = 0;
    int length1 = 0;
    int length2 = 0;
    for (int i=0; i<getLength(); i++) {
        
        //Get the symbols.
        Symbol* symbol1 = getSymbol(i);
        Symbol* symbol2 = comparisonSequence->getSymbol(i);
        
        //If neither one is a gap and they are the same, increment the identity count.
        if (!getAlphabet()->isGap(symbol1) && !comparisonSequence->getAlphabet()->isGap(symbol2) && symbol1->equals(symbol2)) {
            identityCount++;
        }
        
        //Increment the length counters.
        if (!getAlphabet()->isGap(symbol1))
            length1++;
        if (!comparisonSequence->getAlphabet()->isGap(symbol2))
            length2++;
    }
    
    int minLength = (length1<length2)?(length1):(length2);
    return ((float)identityCount)/((float)minLength);
}



















